/*
 * Decompiled with CFR 0.152.
 */
package org.xrpl.xrpl4j.crypto.keys;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.hash.Hashing;
import com.google.common.primitives.UnsignedInteger;
import java.math.BigInteger;
import java.util.Objects;
import java.util.Optional;
import javax.security.auth.Destroyable;
import org.bouncycastle.asn1.sec.SECNamedCurves;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.Ed25519PrivateKeyParameters;
import org.bouncycastle.crypto.params.Ed25519PublicKeyParameters;
import org.xrpl.xrpl4j.codec.addresses.AddressBase58;
import org.xrpl.xrpl4j.codec.addresses.Base58;
import org.xrpl.xrpl4j.codec.addresses.Decoded;
import org.xrpl.xrpl4j.codec.addresses.KeyType;
import org.xrpl.xrpl4j.codec.addresses.SeedCodec;
import org.xrpl.xrpl4j.codec.addresses.UnsignedByte;
import org.xrpl.xrpl4j.codec.addresses.UnsignedByteArray;
import org.xrpl.xrpl4j.codec.addresses.Version;
import org.xrpl.xrpl4j.codec.addresses.exceptions.DecodeException;
import org.xrpl.xrpl4j.crypto.HashingUtils;
import org.xrpl.xrpl4j.crypto.keys.Base58EncodedSecret;
import org.xrpl.xrpl4j.crypto.keys.Entropy;
import org.xrpl.xrpl4j.crypto.keys.KeyPair;
import org.xrpl.xrpl4j.crypto.keys.Passphrase;
import org.xrpl.xrpl4j.crypto.keys.PrivateKey;
import org.xrpl.xrpl4j.crypto.keys.PublicKey;
import org.xrpl.xrpl4j.crypto.signing.bc.Secp256k1;

public interface Seed
extends Destroyable {
    public Decoded decodedSeed();

    public KeyPair deriveKeyPair();

    public static Seed ed25519SeedFromPassphrase(Passphrase passphrase) {
        Objects.requireNonNull(passphrase);
        byte[] entropyBytes = new byte[16];
        Hashing.sha512().hashBytes(passphrase.value()).writeBytesTo(entropyBytes, 0, 16);
        return Seed.ed25519SeedFromEntropy(Entropy.of(entropyBytes));
    }

    public static Seed secp256k1SeedFromPassphrase(Passphrase passphrase) {
        Objects.requireNonNull(passphrase);
        byte[] entropyBytes = new byte[16];
        Hashing.sha512().hashBytes(passphrase.value()).writeBytesTo(entropyBytes, 0, 16);
        return Seed.secp256k1SeedFromEntropy(Entropy.of(entropyBytes));
    }

    public static Seed ed25519Seed() {
        return Seed.ed25519SeedFromEntropy(Entropy.newInstance());
    }

    public static Seed ed25519SeedFromEntropy(Entropy entropy) {
        Objects.requireNonNull(entropy);
        String base58EncodedSeed = AddressBase58.encode(entropy.value(), Lists.newArrayList((Object[])new Version[]{Version.ED25519_SEED}), UnsignedInteger.valueOf((long)entropy.value().length()));
        return new DefaultSeed(UnsignedByteArray.of(AddressBase58.decode(base58EncodedSeed)));
    }

    public static Seed secp256k1Seed() {
        return Seed.secp256k1SeedFromEntropy(Entropy.newInstance());
    }

    public static Seed secp256k1SeedFromEntropy(Entropy entropy) {
        Objects.requireNonNull(entropy);
        String base58EncodedSeed = AddressBase58.encode(entropy.value(), Lists.newArrayList((Object[])new Version[]{Version.FAMILY_SEED}), UnsignedInteger.valueOf((long)entropy.value().length()));
        return new DefaultSeed(UnsignedByteArray.of(Base58.decode(base58EncodedSeed)));
    }

    public static Seed fromBase58EncodedSecret(Base58EncodedSecret base58EncodedSecret) {
        Objects.requireNonNull(base58EncodedSecret);
        return new DefaultSeed(base58EncodedSecret.decodedValueBytes());
    }

    public static class DefaultSeed
    implements Seed {
        private final UnsignedByteArray value;
        private boolean destroyed;

        @VisibleForTesting
        DefaultSeed(UnsignedByteArray value) {
            this.value = Objects.requireNonNull(value);
        }

        @VisibleForTesting
        DefaultSeed(DefaultSeed seed) {
            Objects.requireNonNull(seed);
            this.value = UnsignedByteArray.of(seed.value.toByteArray());
            this.destroyed = seed.isDestroyed();
        }

        @Override
        public Decoded decodedSeed() {
            byte[] copiedByteValue = new byte[this.value.length()];
            System.arraycopy(this.value.toByteArray(), 0, copiedByteValue, 0, this.value.length());
            return SeedCodec.getInstance().decodeSeed(Base58.encode(copiedByteValue));
        }

        @Override
        public KeyPair deriveKeyPair() {
            KeyType type = this.decodedSeed().type().orElseThrow(() -> new IllegalArgumentException("Unsupported seed type."));
            switch (type) {
                case ED25519: {
                    return Ed25519KeyPairService.deriveKeyPair(this);
                }
                case SECP256K1: {
                    return Secp256k1KeyPairService.deriveKeyPair(this);
                }
            }
            throw new IllegalArgumentException("Unsupported seed type.");
        }

        @Override
        public final void destroy() {
            this.value.destroy();
            this.destroyed = true;
        }

        @Override
        public final boolean isDestroyed() {
            return this.destroyed;
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (!(obj instanceof Seed)) {
                return false;
            }
            Seed that = (Seed)obj;
            return that.decodedSeed().equals(this.decodedSeed());
        }

        public int hashCode() {
            return this.value.hashCode();
        }

        public String toString() {
            return "Seed{value=[redacted], destroyed=" + this.destroyed + '}';
        }

        @VisibleForTesting
        static class Ed25519KeyPairService {
            private Ed25519KeyPairService() {
            }

            public static KeyPair deriveKeyPair(Seed seed) {
                Objects.requireNonNull(seed);
                Decoded decoded = seed.decodedSeed();
                if (!decoded.version().equals((Object)Version.ED25519_SEED)) {
                    throw new DecodeException("Seed must use ED25519 algorithm. Algorithm was " + (Object)((Object)decoded.version()));
                }
                UnsignedByteArray rawPrivateKey = HashingUtils.sha512Half(decoded.bytes());
                Ed25519PrivateKeyParameters privateKey = new Ed25519PrivateKeyParameters(rawPrivateKey.toByteArray(), 0);
                Ed25519PublicKeyParameters publicKey = privateKey.generatePublicKey();
                UnsignedByteArray prefixedPublicKey = UnsignedByteArray.of(PublicKey.ED2559_PREFIX, new UnsignedByte[0]).append(UnsignedByteArray.of(publicKey.getEncoded()));
                return KeyPair.builder().privateKey(PrivateKey.fromNaturalBytes(UnsignedByteArray.of(privateKey.getEncoded()), KeyType.ED25519)).publicKey(PublicKey.fromBase16EncodedPublicKey(prefixedPublicKey.hexValue())).build();
            }
        }

        @VisibleForTesting
        static class Secp256k1KeyPairService {
            static X9ECParameters EC_PARAMETERS = SECNamedCurves.getByName((String)"secp256k1");
            static ECDomainParameters EC_DOMAIN_PARAMETERS = new ECDomainParameters(EC_PARAMETERS.getCurve(), EC_PARAMETERS.getG(), EC_PARAMETERS.getN(), EC_PARAMETERS.getH());

            private Secp256k1KeyPairService() {
            }

            public static KeyPair deriveKeyPair(Seed seed) {
                Objects.requireNonNull(seed);
                return Secp256k1KeyPairService.deriveKeyPair(seed.decodedSeed().bytes(), 0);
            }

            private static KeyPair deriveKeyPair(UnsignedByteArray seedBytes, int accountNumber) {
                Objects.requireNonNull(seedBytes);
                BigInteger privateKeyInt = Secp256k1KeyPairService.derivePrivateKey(seedBytes, accountNumber);
                UnsignedByteArray publicKeyByteArray = Secp256k1KeyPairService.derivePublicKey(privateKeyInt);
                Preconditions.checkArgument((publicKeyByteArray.length() == 33 ? 1 : 0) != 0, (Object)("Length was " + publicKeyByteArray.length()));
                return KeyPair.builder().privateKey(PrivateKey.fromPrefixedBytes(Secp256k1.toUnsignedByteArray(privateKeyInt, 33))).publicKey(PublicKey.builder().value(publicKeyByteArray).build()).build();
            }

            private static UnsignedByteArray derivePublicKey(BigInteger privateKey) {
                Objects.requireNonNull(privateKey);
                UnsignedByteArray unpaddedBytes = UnsignedByteArray.of(EC_DOMAIN_PARAMETERS.getG().multiply(privateKey).getEncoded(true));
                return Secp256k1.withZeroPrefixPadding(unpaddedBytes, 33);
            }

            private static BigInteger derivePrivateKey(UnsignedByteArray seed, int accountNumber) {
                Objects.requireNonNull(seed);
                BigInteger privateGen = Secp256k1KeyPairService.deriveScalar(seed);
                if (accountNumber == -1) {
                    return privateGen;
                }
                UnsignedByteArray publicGen = UnsignedByteArray.of(EC_DOMAIN_PARAMETERS.getG().multiply(privateGen).getEncoded(true));
                return Secp256k1KeyPairService.deriveScalar(publicGen, accountNumber).add(privateGen).mod(EC_DOMAIN_PARAMETERS.getN());
            }

            private static BigInteger deriveScalar(UnsignedByteArray seed) {
                Objects.requireNonNull(seed);
                return Secp256k1KeyPairService.deriveScalar(seed, Optional.empty());
            }

            private static BigInteger deriveScalar(UnsignedByteArray seed, Integer discriminator) {
                Objects.requireNonNull(seed);
                Objects.requireNonNull(discriminator);
                return Secp256k1KeyPairService.deriveScalar(seed, Optional.of(discriminator));
            }

            private static BigInteger deriveScalar(UnsignedByteArray seed, Optional<Integer> discriminator) {
                Objects.requireNonNull(seed);
                Objects.requireNonNull(discriminator);
                BigInteger key = null;
                UnsignedByteArray seedCopy = UnsignedByteArray.of(seed.toByteArray());
                for (long i = 0L; i <= 0xFFFFFFFFL; ++i) {
                    discriminator.map(d -> HashingUtils.addUInt32(seedCopy, d));
                    HashingUtils.addUInt32(seedCopy, (int)i);
                    UnsignedByteArray hash = HashingUtils.sha512Half(seedCopy);
                    key = new BigInteger(1, hash.toByteArray());
                    if (key.compareTo(BigInteger.ZERO) > 0 && key.compareTo(EC_DOMAIN_PARAMETERS.getN()) < 0) break;
                }
                return key;
            }
        }
    }
}

