package org.nervos.ckb.sign.signer;

import org.nervos.ckb.crypto.Blake2b;
import org.nervos.ckb.crypto.secp256k1.ECKeyPair;
import org.nervos.ckb.crypto.secp256k1.Sign;
import org.nervos.ckb.sign.Context;
import org.nervos.ckb.sign.ScriptGroup;
import org.nervos.ckb.sign.ScriptSigner;
import org.nervos.ckb.type.Script;
import org.nervos.ckb.type.Transaction;
import org.nervos.ckb.type.WitnessArgs;
import org.nervos.ckb.utils.MoleculeConverter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public class Secp256k1Blake160SighashAllSigner implements ScriptSigner {
  public static boolean signTransactionInPlace(
      Transaction transaction, ScriptGroup scriptGroup, ECKeyPair keyPair) {
    byte[] signature = signTransaction(transaction, scriptGroup, keyPair);
    List<byte[]> witnesses = transaction.witnesses;
    int index = scriptGroup.getInputIndices().get(0);
    WitnessArgs witnessArgs = WitnessArgs.unpack(witnesses.get(index));
    witnessArgs.setLock(signature);
    witnesses.set(index, witnessArgs.pack().toByteArray());
    return true;
  }

  public static byte[] signTransaction(
      Transaction transaction, ScriptGroup scriptGroup, byte[] witnessPlaceholder, ECKeyPair keyPair) {
    byte[] txHash = transaction.computeHash();
    Blake2b blake2b = new Blake2b();
    blake2b.update(txHash);

    blake2b.update(MoleculeConverter.packUint64(witnessPlaceholder.length).toByteArray());
    blake2b.update(witnessPlaceholder);
    List<Integer> includedWitnessIndex = new ArrayList<>();

    for (int i = 1; i < scriptGroup.getInputIndices().size(); i++) {
      includedWitnessIndex.add(scriptGroup.getInputIndices().get(i));
    }
    for (int i = transaction.inputs.size(); i < transaction.witnesses.size(); i++) {
      includedWitnessIndex.add(i);
    }
    for (int i: includedWitnessIndex) {
      byte[] witness = transaction.witnesses.get(i);
      blake2b.update(MoleculeConverter.packUint64(witness.length).toByteArray());
      blake2b.update(witness);
    }

    byte[] message = blake2b.doFinal();
    byte[] signature = Sign.signMessage(message, keyPair).getSignature();
    return signature;
  }

  public static byte[] signTransaction(
      Transaction transaction, ScriptGroup scriptGroup, ECKeyPair keyPair) {
    byte[] witnessPlaceholder = transaction.witnesses.get(scriptGroup.getInputIndices().get(0));
    return signTransaction(transaction, scriptGroup, witnessPlaceholder, keyPair);
  }

  @Override
  public boolean signTransaction(
      Transaction transaction, ScriptGroup scriptGroup, Context context) {
    Script script = scriptGroup.getScript();
    ECKeyPair keyPair = context.getKeyPair();
    if (isMatched(keyPair, script.args)) {
      return signTransactionInPlace(transaction, scriptGroup, keyPair);
    } else {
      return false;
    }
  }

  // Check if the script with `scriptArgs` is generated by and can be unlocked by `privateKey`
  public static boolean isMatched(ECKeyPair keyPair, byte[] scriptArgs) {
    if (scriptArgs == null || keyPair == null) {
      return false;
    }
    byte[] hash = Blake2b.digest(keyPair.getEncodedPublicKey(true));
    hash = Arrays.copyOfRange(hash, 0, 20);
    return Arrays.equals(scriptArgs, hash);
  }
}
