/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.cli;

import java.io.PrintStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DeprecatedAttributes;
import org.apache.commons.cli.Option;
import org.apache.solr.cli.CLIO;
import org.apache.solr.cli.SolrCLI;
import org.apache.solr.cli.SolrProcessManager;
import org.apache.solr.cli.ToolBase;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.request.CollectionAdminRequest;
import org.apache.solr.client.solrj.request.GenericSolrRequest;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.common.util.URLUtil;
import org.noggit.CharArr;
import org.noggit.JSONWriter;

public class StatusTool
extends ToolBase {
    private final SolrProcessManager processMgr = new SolrProcessManager();
    private static final Option OPTION_MAXWAITSECS = Option.builder().longOpt("max-wait-secs").argName("SECS").hasArg().required(false).deprecated().desc("Wait up to the specified number of seconds to see Solr running.").build();
    private static final Option OPTION_MAXWAITSECS_DEPRECATED = Option.builder((String)"maxWaitSecs").argName("SECS").hasArg().required(false).desc("Wait up to the specified number of seconds to see Solr running.").deprecated(DeprecatedAttributes.builder().setForRemoval(true).setSince("9.7").setDescription("Use --max-wait-secs instead").get()).build();
    public static final Option OPTION_PORT = Option.builder((String)"p").longOpt("port").argName("PORT").required(false).hasArg().desc("Port on localhost to check status for").build();
    public static final Option OPTION_SHORT = Option.builder().longOpt("short").argName("SHORT").required(false).desc("Short format. Prints one URL per line for running instances").build();

    public StatusTool() {
        this(CLIO.getOutStream());
    }

    public StatusTool(PrintStream stdout) {
        super(stdout);
    }

    @Override
    public String getName() {
        return "status";
    }

    @Override
    public List<Option> getOptions() {
        return List.of(SolrCLI.OPTION_SOLRURL, OPTION_MAXWAITSECS, OPTION_MAXWAITSECS_DEPRECATED, OPTION_PORT, OPTION_SHORT);
    }

    @Override
    public void runImpl(CommandLine cli) throws Exception {
        Collection<SolrProcessManager.SolrProcess> procs;
        String solrUrl = cli.getOptionValue(SolrCLI.OPTION_SOLRURL);
        Integer port = cli.hasOption(OPTION_PORT) ? Integer.valueOf(Integer.parseInt(cli.getOptionValue(OPTION_PORT))) : null;
        boolean shortFormat = cli.hasOption(OPTION_SHORT);
        int maxWaitSecs = Integer.parseInt(cli.getOptionValue("max-wait-secs", "0"));
        if (port != null && solrUrl != null) {
            throw new IllegalArgumentException("Only one of port or url can be specified");
        }
        if (solrUrl != null) {
            if (!URLUtil.hasScheme((String)solrUrl)) {
                CLIO.err("Invalid URL provided: " + solrUrl);
                System.exit(1);
            }
            if (maxWaitSecs > 0) {
                try {
                    this.waitForSolrUpAndPrintStatus(solrUrl, cli, maxWaitSecs);
                    System.exit(0);
                }
                catch (Exception e) {
                    CLIO.err(e.getMessage());
                    System.exit(1);
                }
            } else {
                boolean running = this.printStatusFromRunningSolr(solrUrl, cli);
                System.exit(running ? 0 : 1);
            }
        }
        if (port != null) {
            Optional<SolrProcessManager.SolrProcess> proc = this.processMgr.processForPort(port);
            if (proc.isEmpty()) {
                CLIO.err("Could not find a running Solr on port " + port);
                System.exit(1);
            } else {
                solrUrl = proc.get().getLocalUrl();
                if (shortFormat) {
                    CLIO.out(solrUrl);
                } else {
                    this.printProcessStatus(proc.get(), cli);
                }
                System.exit(0);
            }
        }
        if (!(procs = this.processMgr.scanSolrPidFiles()).isEmpty()) {
            for (SolrProcessManager.SolrProcess process : procs) {
                if (shortFormat) {
                    CLIO.out(process.getLocalUrl());
                    continue;
                }
                this.printProcessStatus(process, cli);
            }
        } else if (!shortFormat) {
            CLIO.out("\nNo Solr nodes are running.\n");
        }
    }

    private void printProcessStatus(SolrProcessManager.SolrProcess process, CommandLine cli) throws Exception {
        int maxWaitSecs = Integer.parseInt(SolrCLI.getOptionWithDeprecatedAndDefault(cli, "max-wait-secs", "maxWaitSecs", "0"));
        boolean shortFormat = cli.hasOption(OPTION_SHORT);
        String pidUrl = process.getLocalUrl();
        if (shortFormat) {
            CLIO.out(pidUrl);
        } else if (maxWaitSecs > 0) {
            this.waitForSolrUpAndPrintStatus(pidUrl, cli, maxWaitSecs);
        } else {
            CLIO.out(String.format(Locale.ROOT, "\nSolr process %s running on port %s", process.getPid(), process.getPort()));
            this.printStatusFromRunningSolr(pidUrl, cli);
        }
        CLIO.out("");
    }

    private Integer portFromUrl(String solrUrl) {
        try {
            URI uri = new URI(solrUrl);
            int port = uri.getPort();
            if (port == -1) {
                return uri.getScheme().equals("https") ? 443 : 80;
            }
            return port;
        }
        catch (URISyntaxException e) {
            CLIO.err("Invalid URL provided, does not contain port");
            System.exit(1);
            return null;
        }
    }

    public void waitForSolrUpAndPrintStatus(String solrUrl, CommandLine cli, int maxWaitSecs) throws Exception {
        int solrPort = this.portFromUrl(solrUrl);
        this.echo("Waiting up to " + maxWaitSecs + " seconds to see Solr running on port " + solrPort);
        boolean solrUp = this.waitForSolrUp(solrUrl, cli, maxWaitSecs);
        if (!solrUp) {
            throw new Exception("Solr at " + solrUrl + " did not come online within " + maxWaitSecs + " seconds!");
        }
        this.echo("Started Solr server on port " + solrPort + ". Happy searching!");
    }

    public boolean waitForSolrUp(String solrUrl, CommandLine cli, int maxWaitSecs) throws Exception {
        try {
            this.waitToSeeSolrUp(solrUrl, maxWaitSecs, TimeUnit.SECONDS);
            return true;
        }
        catch (TimeoutException timeout) {
            return false;
        }
    }

    public boolean printStatusFromRunningSolr(String solrUrl, CommandLine cli) throws Exception {
        String statusJson = null;
        try {
            statusJson = this.statusFromRunningSolr(solrUrl, cli);
        }
        catch (Exception exception) {
            // empty catch block
        }
        if (statusJson != null) {
            CLIO.out(statusJson);
        } else {
            CLIO.err("Solr at " + solrUrl + " not online.");
        }
        return statusJson != null;
    }

    public String statusFromRunningSolr(String solrUrl, CommandLine cli) throws Exception {
        try {
            CharArr arr = new CharArr();
            new JSONWriter(arr, 2).write(this.getStatus(solrUrl));
            return arr.toString();
        }
        catch (Exception exc) {
            if (SolrCLI.exceptionIsAuthRelated(exc)) {
                throw exc;
            }
            if (SolrCLI.checkCommunicationError(exc)) {
                return null;
            }
            throw new Exception("Failed to get system information from " + solrUrl + " due to: " + exc);
        }
    }

    public Map<String, Object> waitToSeeSolrUp(String solrUrl, long maxWait, TimeUnit unit) throws Exception {
        long timeout = System.nanoTime() + TimeUnit.NANOSECONDS.convert(maxWait, unit);
        while (System.nanoTime() < timeout) {
            try {
                return this.getStatus(solrUrl);
            }
            catch (Exception exc) {
                if (SolrCLI.exceptionIsAuthRelated(exc)) {
                    throw exc;
                }
                try {
                    Thread.sleep(2000L);
                }
                catch (InterruptedException interrupted) {
                    timeout = 0L;
                }
            }
        }
        throw new TimeoutException("Did not see Solr at " + solrUrl + " come online within " + TimeUnit.SECONDS.convert(maxWait, unit) + " seconds!");
    }

    public Map<String, Object> getStatus(String solrUrl) throws Exception {
        Map<String, Object> status;
        if (!((String)solrUrl).endsWith("/")) {
            solrUrl = (String)solrUrl + "/";
        }
        try (SolrClient solrClient = SolrCLI.getSolrClient((String)solrUrl);){
            NamedList systemInfo = solrClient.request((SolrRequest)new GenericSolrRequest(SolrRequest.METHOD.GET, "/admin/info/system"));
            status = this.reportStatus((NamedList<Object>)systemInfo, solrClient);
        }
        return status;
    }

    public Map<String, Object> reportStatus(NamedList<Object> info, SolrClient solrClient) throws Exception {
        LinkedHashMap<String, Object> status = new LinkedHashMap<String, Object>();
        String solrHome = (String)info.get("solr_home");
        status.put("solr_home", solrHome != null ? solrHome : "?");
        status.put("version", info.findRecursive(new String[]{"lucene", "solr-impl-version"}));
        status.put("startTime", info.findRecursive(new String[]{"jvm", "jmx", "startTime"}).toString());
        status.put("uptime", SolrCLI.uptime((Long)info.findRecursive(new String[]{"jvm", "jmx", "upTimeMS"})));
        String usedMemory = (String)info.findRecursive(new String[]{"jvm", "memory", "used"});
        String totalMemory = (String)info.findRecursive(new String[]{"jvm", "memory", "total"});
        status.put("memory", usedMemory + " of " + totalMemory);
        if ("solrcloud".equals(info.get("mode"))) {
            String zkHost = (String)info.get("zkHost");
            status.put("cloud", this.getCloudStatus(solrClient, zkHost));
        }
        return status;
    }

    protected Map<String, String> getCloudStatus(SolrClient solrClient, String zkHost) throws Exception {
        LinkedHashMap<String, String> cloudStatus = new LinkedHashMap<String, String>();
        cloudStatus.put("ZooKeeper", zkHost != null ? zkHost : "?");
        NamedList json = solrClient.request((SolrRequest)new CollectionAdminRequest.ClusterStatus());
        List liveNodes = (List)json.findRecursive(new String[]{"cluster", "live_nodes"});
        cloudStatus.put("liveNodes", String.valueOf(liveNodes.size()));
        NamedList collections = (NamedList)json.findRecursive(new String[]{"cluster", "collections"});
        cloudStatus.put("collections", String.valueOf(collections.size()));
        return cloudStatus;
    }
}

