/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.core;

import com.google.common.collect.Lists;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.lang.invoke.MethodHandles;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.stream.Collectors;
import org.apache.solr.common.SolrException;
import org.apache.solr.core.CoreContainer;
import org.apache.solr.core.CoreDescriptor;
import org.apache.solr.core.CoresLocator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CorePropertiesLocator
implements CoresLocator {
    public static final String PROPERTIES_FILENAME = "core.properties";
    private static final Logger logger = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private final Path rootDirectory;

    public CorePropertiesLocator(Path coreDiscoveryRoot) {
        this.rootDirectory = coreDiscoveryRoot;
        logger.debug("Config-defined core root directory: {}", (Object)this.rootDirectory);
    }

    @Override
    public void create(CoreContainer cc, CoreDescriptor ... coreDescriptors) {
        for (CoreDescriptor cd : coreDescriptors) {
            Path propertiesFile = this.rootDirectory.resolve(cd.getInstanceDir()).resolve(PROPERTIES_FILENAME);
            if (Files.exists(propertiesFile, new LinkOption[0])) {
                throw new SolrException(SolrException.ErrorCode.BAD_REQUEST, "Could not create a new core in " + cd.getInstanceDir() + "as another core is already defined there");
            }
            this.writePropertiesFile(cd, propertiesFile);
        }
    }

    @Override
    public void persist(CoreContainer cc, CoreDescriptor ... coreDescriptors) {
        for (CoreDescriptor cd : coreDescriptors) {
            Path propFile = this.rootDirectory.resolve(cd.getInstanceDir()).resolve(PROPERTIES_FILENAME);
            this.writePropertiesFile(cd, propFile);
        }
    }

    private void writePropertiesFile(CoreDescriptor cd, Path propfile) {
        Properties p = this.buildCoreProperties(cd);
        try {
            Files.createDirectories(propfile.getParent(), new FileAttribute[0]);
            try (OutputStreamWriter os = new OutputStreamWriter(Files.newOutputStream(propfile, new OpenOption[0]), StandardCharsets.UTF_8);){
                p.store(os, "Written by CorePropertiesLocator");
            }
        }
        catch (IOException e) {
            logger.error("Couldn't persist core properties to {}: {}", (Object)propfile, (Object)e.getMessage());
        }
    }

    @Override
    public void delete(CoreContainer cc, CoreDescriptor ... coreDescriptors) {
        if (coreDescriptors == null) {
            return;
        }
        for (CoreDescriptor cd : coreDescriptors) {
            if (cd == null) continue;
            Path propfile = this.rootDirectory.resolve(cd.getInstanceDir()).resolve(PROPERTIES_FILENAME);
            try {
                Files.deleteIfExists(propfile);
            }
            catch (IOException e) {
                logger.warn("Couldn't delete core properties file {}: {}", (Object)propfile, (Object)e.getMessage());
            }
        }
    }

    @Override
    public void rename(CoreContainer cc, CoreDescriptor oldCD, CoreDescriptor newCD) {
        this.persist(cc, newCD);
    }

    @Override
    public void swap(CoreContainer cc, CoreDescriptor cd1, CoreDescriptor cd2) {
        this.persist(cc, cd1, cd2);
    }

    @Override
    public List<CoreDescriptor> discover(final CoreContainer cc) {
        logger.debug("Looking for core definitions underneath {}", (Object)this.rootDirectory);
        final ArrayList cds = Lists.newArrayList();
        try {
            HashSet<FileVisitOption> options = new HashSet<FileVisitOption>();
            options.add(FileVisitOption.FOLLOW_LINKS);
            int maxDepth = 256;
            Files.walkFileTree(this.rootDirectory, options, 256, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                    if (file.getFileName().toString().equals(CorePropertiesLocator.PROPERTIES_FILENAME)) {
                        CoreDescriptor cd = CorePropertiesLocator.this.buildCoreDescriptor(file, cc);
                        if (cd != null) {
                            logger.debug("Found core {} in {}", (Object)cd.getName(), (Object)cd.getInstanceDir());
                            cds.add(cd);
                        }
                        return FileVisitResult.SKIP_SIBLINGS;
                    }
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                    if (file.equals(CorePropertiesLocator.this.rootDirectory)) {
                        logger.error("Error reading core root directory {}: {}", (Object)file, (Object)exc);
                        throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Error reading core root directory");
                    }
                    logger.warn("Error visiting {}: {}", (Object)file, (Object)exc);
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Couldn't walk file tree under " + this.rootDirectory, (Throwable)e);
        }
        logger.info("Found {} core definitions underneath {}", (Object)cds.size(), (Object)this.rootDirectory);
        if (cds.size() > 0) {
            logger.info("Cores are: {}", cds.stream().map(CoreDescriptor::getName).collect(Collectors.toList()));
        }
        return cds;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    protected CoreDescriptor buildCoreDescriptor(Path propertiesFile, CoreContainer cc) {
        Path instanceDir = propertiesFile.getParent();
        Properties coreProperties = new Properties();
        try (InputStream fis = Files.newInputStream(propertiesFile, new OpenOption[0]);){
            coreProperties.load(new InputStreamReader(fis, StandardCharsets.UTF_8));
            String name = CorePropertiesLocator.createName(coreProperties, instanceDir);
            HashMap<String, String> propMap = new HashMap<String, String>();
            for (String key : coreProperties.stringPropertyNames()) {
                propMap.put(key, coreProperties.getProperty(key));
            }
            CoreDescriptor ret = new CoreDescriptor(name, instanceDir, propMap, cc.getContainerProperties(), cc.isZooKeeperAware());
            ret.loadExtraProperties();
            CoreDescriptor coreDescriptor = ret;
            return coreDescriptor;
        }
        catch (IOException e) {
            logger.error("Couldn't load core descriptor from {}:{}", (Object)propertiesFile, (Object)e.toString());
            return null;
        }
    }

    protected static String createName(Properties p, Path instanceDir) {
        return p.getProperty("name", instanceDir.getFileName().toString());
    }

    protected Properties buildCoreProperties(CoreDescriptor cd) {
        Properties p = new Properties();
        p.putAll((Map<?, ?>)cd.getPersistableStandardProperties());
        p.putAll((Map<?, ?>)cd.getPersistableUserProperties());
        return p;
    }
}

