/*
 * Decompiled with CFR 0.152.
 */
package one.block.eosiojava.utilities;

import com.google.common.base.Preconditions;
import com.google.common.primitives.Bytes;
import java.io.CharArrayReader;
import java.io.Reader;
import java.math.BigInteger;
import java.util.Arrays;
import one.block.eosiojava.enums.AlgorithmEmployed;
import one.block.eosiojava.error.utilities.Base58ManipulationError;
import one.block.eosiojava.error.utilities.DerToPemConversionError;
import one.block.eosiojava.error.utilities.EOSFormatterError;
import one.block.eosiojava.error.utilities.EosFormatterSignatureIsNotCanonicalError;
import one.block.eosiojava.error.utilities.LowSVerificationError;
import one.block.eosiojava.utilities.PEMProcessor;
import org.bitcoinj.core.Base58;
import org.bitcoinj.core.Sha256Hash;
import org.bitcoinj.core.Utils;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1Integer;
import org.bouncycastle.asn1.DLSequence;
import org.bouncycastle.asn1.sec.SECNamedCurves;
import org.bouncycastle.asn1.x9.X9ECParameters;
import org.bouncycastle.asn1.x9.X9IntegerConverter;
import org.bouncycastle.crypto.digests.RIPEMD160Digest;
import org.bouncycastle.crypto.ec.CustomNamedCurves;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.spec.ECNamedCurveParameterSpec;
import org.bouncycastle.math.ec.ECAlgorithms;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.math.ec.FixedPointUtil;
import org.bouncycastle.util.encoders.Base64;
import org.bouncycastle.util.encoders.Hex;
import org.bouncycastle.util.io.pem.PemObject;
import org.bouncycastle.util.io.pem.PemReader;
import org.jetbrains.annotations.NotNull;

public class EOSFormatter {
    private static final String PATTERN_STRING_EOS_PREFIX_EOS = "EOS";
    private static final String PATTERN_STRING_EOS_PREFIX_PUB_R1 = "PUB_R1_";
    private static final String PATTERN_STRING_EOS_PREFIX_PUB_K1 = "PUB_K1_";
    private static final String PATTERN_STRING_EOS_PREFIX_PVT_R1 = "PVT_R1_";
    private static final String PATTERN_STRING_EOS_PREFIX_SIG_R1 = "SIG_R1_";
    private static final String PATTERN_STRING_EOS_PREFIX_SIG_K1 = "SIG_K1_";
    private static final String PATTERN_STRING_PEM_PREFIX_PRIVATE_KEY_SECP256R1 = "30770201010420";
    private static final String PATTERN_STRING_PEM_PREFIX_PRIVATE_KEY_SECP256K1 = "302E0201010420";
    private static final String PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256R1_UNCOMPRESSED = "3059301306072a8648ce3d020106082a8648ce3d030107034200";
    private static final String PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256K1_UNCOMPRESSED = "3056301006072a8648ce3d020106052b8104000a034200";
    private static final String PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256R1_COMPRESSED = "3039301306072a8648ce3d020106082a8648ce3d030107032200";
    private static final String PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256K1_COMPRESSED = "3036301006072a8648ce3d020106052b8104000a032200";
    private static final String PATTERN_STRING_PEM_SUFFIX_PRIVATE_KEY_SECP256K1 = "A00706052B8104000A";
    private static final String PATTERN_STRING_PEM_SUFFIX_PRIVATE_KEY_SECP256R1 = "A00A06082A8648CE3D030107";
    private static final String PEM_HEADER_PUBLIC_KEY = "-----BEGIN PUBLIC KEY-----";
    private static final String PEM_FOOTER_PUBLIC_KEY = "-----END PUBLIC KEY-----";
    private static final String PEM_HEADER_PRIVATE_KEY = "-----BEGIN EC PRIVATE KEY-----";
    private static final String PEM_FOOTER_PRIVATE_KEY = "-----END EC PRIVATE KEY-----";
    private static final String PEM_HEADER_EC_PRIVATE_KEY = "EC PRIVATE KEY";
    private static final String PEM_HEADER_EC_PUBLIC_KEY = "PUBLIC KEY";
    private static final String SECP256R1_AND_PRIME256V1_CHECKSUM_VALIDATION_SUFFIX = "R1";
    private static final String SECP256K1_CHECKSUM_VALIDATION_SUFFIX = "K1";
    private static final String LEGACY_CHECKSUM_VALIDATION_SUFFIX = "";
    private static final int STANDARD_KEY_LENGTH = 32;
    private static final int CHECKSUM_BYTES = 4;
    private static final int FIRST_TWO_BYTES_OF_KEY = 4;
    private static final int DATA_SEQUENCE_LENGTH_BYTE_POSITION = 2;
    private static final int EOS_SECP256K1_HEADER_BYTE = 128;
    private static final byte UNCOMPRESSED_PUBLIC_KEY_BYTE_INDICATOR = 4;
    private static final byte COMPRESSED_PUBLIC_KEY_BYTE_INDICATOR_POSITIVE_Y = 2;
    private static final byte COMPRESSED_PUBLIC_KEY_BYTE_INDICATOR_NEGATIVE_Y = 3;
    private static final int CHAIN_ID_LENGTH = 64;
    private static final int CONTEXT_FREE_DATA_LENGTH = 64;
    private static final int MINIMUM_SIGNABLE_TRANSACTION_LENGTH = 129;
    private static final int VALUE_TO_ADD_TO_SIGNATURE_HEADER = 31;
    private static final int EXPECTED_R_OR_S_LENGTH = 32;
    private static final int NUMBER_OF_POSSIBLE_PUBLIC_KEYS = 4;
    private static final String SECP256_R1 = "secp256r1";
    private static final String SECP256_K1 = "secp256k1";
    private static final ECDomainParameters ecParamsR1;
    private static final ECDomainParameters ecParamsK1;
    private static final X9ECParameters CURVE_PARAMS_R1;
    private static final X9ECParameters CURVE_PARAMS_K1;
    private static final ECDomainParameters CURVE_R1;
    private static final BigInteger HALF_CURVE_ORDER_R1;
    private static final ECDomainParameters CURVE_K1;
    private static final BigInteger HALF_CURVE_ORDER_K1;

    @NotNull
    public static String convertPEMFormattedPublicKeyToEOSFormat(@NotNull String publicKeyPEM, boolean requireLegacyFormOfSecp256k1Key) throws EOSFormatterError {
        String type;
        PemObject pemObject;
        String eosFormattedPublicKey = publicKeyPEM;
        try (CharArrayReader reader = new CharArrayReader(eosFormattedPublicKey.toCharArray());
             PemReader pemReader = new PemReader((Reader)reader);){
            pemObject = pemReader.readPemObject();
            type = pemObject.getType();
        }
        catch (Exception e) {
            throw new EOSFormatterError("This is not a PEM formatted private key!", e);
        }
        if (type.matches("(?i:.*PUBLIC KEY.*)")) {
            AlgorithmEmployed algorithmEmployed;
            eosFormattedPublicKey = Hex.toHexString((byte[])pemObject.getContent());
            if (eosFormattedPublicKey.toUpperCase().contains(PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256R1_UNCOMPRESSED.toUpperCase())) {
                eosFormattedPublicKey = eosFormattedPublicKey.toUpperCase().replace(PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256R1_UNCOMPRESSED.toUpperCase(), LEGACY_CHECKSUM_VALIDATION_SUFFIX);
                algorithmEmployed = AlgorithmEmployed.SECP256R1;
            } else if (eosFormattedPublicKey.toUpperCase().contains(PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256R1_COMPRESSED.toUpperCase())) {
                eosFormattedPublicKey = eosFormattedPublicKey.toUpperCase().replace(PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256R1_COMPRESSED.toUpperCase(), LEGACY_CHECKSUM_VALIDATION_SUFFIX);
                algorithmEmployed = AlgorithmEmployed.SECP256R1;
            } else if (eosFormattedPublicKey.toUpperCase().contains(PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256K1_UNCOMPRESSED.toUpperCase())) {
                eosFormattedPublicKey = eosFormattedPublicKey.toUpperCase().replace(PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256K1_UNCOMPRESSED.toUpperCase(), LEGACY_CHECKSUM_VALIDATION_SUFFIX);
                algorithmEmployed = AlgorithmEmployed.SECP256K1;
            } else if (eosFormattedPublicKey.toUpperCase().contains(PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256K1_COMPRESSED.toUpperCase())) {
                eosFormattedPublicKey = eosFormattedPublicKey.toUpperCase().replace(PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256K1_COMPRESSED.toUpperCase(), LEGACY_CHECKSUM_VALIDATION_SUFFIX);
                algorithmEmployed = AlgorithmEmployed.SECP256K1;
            } else {
                throw new EOSFormatterError("DER format of private key is incorrect!");
            }
            byte[] eosFormattedPublicKeyBytes = Hex.decode((String)eosFormattedPublicKey);
            if (eosFormattedPublicKeyBytes[0] == 4) {
                try {
                    eosFormattedPublicKey = Hex.toHexString((byte[])EOSFormatter.compressPublickey(Hex.decode((String)eosFormattedPublicKey), algorithmEmployed));
                }
                catch (Exception e) {
                    throw new EOSFormatterError(e);
                }
            }
            try {
                eosFormattedPublicKey = EOSFormatter.encodePublicKey(Hex.decode((String)eosFormattedPublicKey), algorithmEmployed, requireLegacyFormOfSecp256k1Key);
            }
            catch (Base58ManipulationError e) {
                throw new EOSFormatterError(e);
            }
        }
        throw new EOSFormatterError("This is not a PEM formatted private key!");
        return eosFormattedPublicKey;
    }

    @NotNull
    public static String convertEOSPublicKeyToPEMFormat(@NotNull String publicKeyEOS) throws EOSFormatterError {
        byte[] base58DecodedPublicKey;
        String keyPrefix;
        AlgorithmEmployed algorithmEmployed;
        String pemFormattedPublickKey = publicKeyEOS;
        if (pemFormattedPublickKey.toUpperCase().contains(PATTERN_STRING_EOS_PREFIX_PUB_R1.toUpperCase())) {
            algorithmEmployed = AlgorithmEmployed.SECP256R1;
            keyPrefix = PATTERN_STRING_EOS_PREFIX_PUB_R1;
            pemFormattedPublickKey = pemFormattedPublickKey.replace(PATTERN_STRING_EOS_PREFIX_PUB_R1, LEGACY_CHECKSUM_VALIDATION_SUFFIX);
        } else if (pemFormattedPublickKey.toUpperCase().contains(PATTERN_STRING_EOS_PREFIX_PUB_K1.toUpperCase())) {
            algorithmEmployed = AlgorithmEmployed.SECP256K1;
            keyPrefix = PATTERN_STRING_EOS_PREFIX_PUB_K1;
            pemFormattedPublickKey = pemFormattedPublickKey.replace(PATTERN_STRING_EOS_PREFIX_PUB_K1, LEGACY_CHECKSUM_VALIDATION_SUFFIX);
        } else if (pemFormattedPublickKey.toUpperCase().contains(PATTERN_STRING_EOS_PREFIX_EOS.toUpperCase())) {
            algorithmEmployed = AlgorithmEmployed.SECP256K1;
            keyPrefix = PATTERN_STRING_EOS_PREFIX_EOS;
            pemFormattedPublickKey = pemFormattedPublickKey.replace(PATTERN_STRING_EOS_PREFIX_EOS, LEGACY_CHECKSUM_VALIDATION_SUFFIX);
        } else {
            throw new EOSFormatterError("The EOS public key provided is invalid!");
        }
        try {
            base58DecodedPublicKey = EOSFormatter.decodePublicKey(pemFormattedPublickKey, keyPrefix);
        }
        catch (Exception e) {
            throw new EOSFormatterError("An error occurred while Base58 decoding the EOS key!", e);
        }
        pemFormattedPublickKey = Hex.toHexString((byte[])base58DecodedPublicKey);
        if (base58DecodedPublicKey[0] == 4) {
            try {
                pemFormattedPublickKey = Hex.toHexString((byte[])EOSFormatter.compressPublickey(Hex.decode((String)pemFormattedPublickKey), algorithmEmployed));
            }
            catch (Exception e) {
                throw new EOSFormatterError(e);
            }
        }
        switch (algorithmEmployed) {
            case SECP256R1: {
                pemFormattedPublickKey = PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256R1_COMPRESSED + pemFormattedPublickKey;
                break;
            }
            case SECP256K1: {
                pemFormattedPublickKey = PATTERN_STRING_PEM_PREFIX_PUBLIC_KEY_SECP256K1_COMPRESSED + pemFormattedPublickKey;
                break;
            }
            default: {
                throw new EOSFormatterError("Unsupported algorithm!");
            }
        }
        if (pemFormattedPublickKey.length() <= 4) {
            throw new EOSFormatterError("The EOS public key provided is invalid!");
        }
        int i = (pemFormattedPublickKey.length() - 4) / 2;
        String correctedLength = Integer.toHexString(i);
        pemFormattedPublickKey = pemFormattedPublickKey.substring(0, 2) + correctedLength + pemFormattedPublickKey.substring(4);
        try {
            pemFormattedPublickKey = EOSFormatter.derToPEM(Hex.decode((String)pemFormattedPublickKey), PEMObjectType.PUBLICKEY);
        }
        catch (Exception e) {
            throw new EOSFormatterError(e);
        }
        return pemFormattedPublickKey;
    }

    @NotNull
    public static String convertDERSignatureToEOSFormat(@NotNull byte[] signatureDER, @NotNull byte[] signableTransaction, @NotNull String publicKeyPEM) throws EOSFormatterError {
        String eosFormattedSignature = LEGACY_CHECKSUM_VALIDATION_SUFFIX;
        try (ASN1InputStream asn1InputStream = new ASN1InputStream(signatureDER);){
            String signaturePrefix;
            byte[] signatureWithCheckSum;
            PEMProcessor publicKey = new PEMProcessor(publicKeyPEM);
            AlgorithmEmployed algorithmEmployed = publicKey.getAlgorithm();
            byte[] keyData = publicKey.getKeyData();
            DLSequence sequence = (DLSequence)asn1InputStream.readObject();
            BigInteger r = ((ASN1Integer)sequence.getObjectAt(0)).getPositiveValue();
            BigInteger s = ((ASN1Integer)sequence.getObjectAt(1)).getPositiveValue();
            s = EOSFormatter.checkAndHandleLowS(s, algorithmEmployed);
            int recoverId = EOSFormatter.getRecoveryId(r, s, Sha256Hash.of((byte[])signableTransaction), keyData, algorithmEmployed);
            if (recoverId < 0) {
                throw new IllegalStateException("Could not recover public key from Signature.");
            }
            byte headerByte = Integer.valueOf(recoverId += 31).byteValue();
            byte[] decodedSignature = Bytes.concat((byte[][])new byte[][]{{headerByte}, Utils.bigIntegerToBytes((BigInteger)r, (int)32), Utils.bigIntegerToBytes((BigInteger)s, (int)32)});
            if (algorithmEmployed.equals((Object)AlgorithmEmployed.SECP256K1) && !EOSFormatter.isCanonical(decodedSignature)) {
                throw new IllegalArgumentException("Input signature is not canonical.");
            }
            switch (algorithmEmployed) {
                case SECP256R1: {
                    signatureWithCheckSum = EOSFormatter.addCheckSumToSignature(decodedSignature, SECP256R1_AND_PRIME256V1_CHECKSUM_VALIDATION_SUFFIX.getBytes());
                    signaturePrefix = PATTERN_STRING_EOS_PREFIX_SIG_R1;
                    break;
                }
                case SECP256K1: {
                    signatureWithCheckSum = EOSFormatter.addCheckSumToSignature(decodedSignature, SECP256K1_CHECKSUM_VALIDATION_SUFFIX.getBytes());
                    signaturePrefix = PATTERN_STRING_EOS_PREFIX_SIG_K1;
                    break;
                }
                default: {
                    throw new EOSFormatterError("Unsupported algorithm!");
                }
            }
            eosFormattedSignature = signaturePrefix.concat(Base58.encode((byte[])signatureWithCheckSum));
        }
        catch (Exception e) {
            throw new EOSFormatterError("An error occured formating the signature!", e);
        }
        return eosFormattedSignature;
    }

    @NotNull
    public static String convertRawRandSofSignatureToEOSFormat(@NotNull String signatureR, String signatureS, @NotNull byte[] signableTransaction, @NotNull String publicKeyPEM) throws EOSFormatterError {
        String eosFormattedSignature = LEGACY_CHECKSUM_VALIDATION_SUFFIX;
        try {
            String signaturePrefix;
            byte[] signatureWithCheckSum;
            PEMProcessor publicKey = new PEMProcessor(publicKeyPEM);
            AlgorithmEmployed algorithmEmployed = publicKey.getAlgorithm();
            byte[] keyData = publicKey.getKeyData();
            BigInteger r = new BigInteger(signatureR);
            BigInteger s = new BigInteger(signatureS);
            s = EOSFormatter.checkAndHandleLowS(s, algorithmEmployed);
            int recoverId = EOSFormatter.getRecoveryId(r, s, Sha256Hash.of((byte[])signableTransaction), keyData, algorithmEmployed);
            if (recoverId < 0) {
                throw new IllegalStateException("Could not recover public key from Signature.");
            }
            byte headerByte = Integer.valueOf(recoverId += 31).byteValue();
            byte[] decodedSignature = Bytes.concat((byte[][])new byte[][]{{headerByte}, Utils.bigIntegerToBytes((BigInteger)r, (int)32), Utils.bigIntegerToBytes((BigInteger)s, (int)32)});
            if (algorithmEmployed.equals((Object)AlgorithmEmployed.SECP256K1) && !EOSFormatter.isCanonical(decodedSignature)) {
                throw new EosFormatterSignatureIsNotCanonicalError("Input signature is not canonical.");
            }
            switch (algorithmEmployed) {
                case SECP256R1: {
                    signatureWithCheckSum = EOSFormatter.addCheckSumToSignature(decodedSignature, SECP256R1_AND_PRIME256V1_CHECKSUM_VALIDATION_SUFFIX.getBytes());
                    signaturePrefix = PATTERN_STRING_EOS_PREFIX_SIG_R1;
                    break;
                }
                case SECP256K1: {
                    signatureWithCheckSum = EOSFormatter.addCheckSumToSignature(decodedSignature, SECP256K1_CHECKSUM_VALIDATION_SUFFIX.getBytes());
                    signaturePrefix = PATTERN_STRING_EOS_PREFIX_SIG_K1;
                    break;
                }
                default: {
                    throw new EOSFormatterError("Unsupported algorithm!");
                }
            }
            eosFormattedSignature = signaturePrefix.concat(Base58.encode((byte[])signatureWithCheckSum));
        }
        catch (Exception e) {
            throw new EOSFormatterError("An error occured formating the signature!", e);
        }
        return eosFormattedSignature;
    }

    @NotNull
    public static String convertPEMFormattedPrivateKeyToEOSFormat(@NotNull String privateKeyPEM) throws EOSFormatterError {
        StringBuilder builder;
        String type;
        PemObject pemObject;
        String eosFormattedPrivateKey = privateKeyPEM;
        try (CharArrayReader reader = new CharArrayReader(eosFormattedPrivateKey.toCharArray());
             PemReader pemReader = new PemReader((Reader)reader);){
            pemObject = pemReader.readPemObject();
            type = pemObject.getType();
        }
        catch (Exception e) {
            throw new EOSFormatterError("This is not a PEM formatted private key!", e);
        }
        if (type.matches("(?i:.*EC PRIVATE KEY.*)")) {
            AlgorithmEmployed algorithmEmployed;
            eosFormattedPrivateKey = Hex.toHexString((byte[])pemObject.getContent());
            if (eosFormattedPrivateKey.matches("(?i:.*A00A06082A8648CE3D030107.*)")) {
                algorithmEmployed = AlgorithmEmployed.SECP256R1;
            } else if (eosFormattedPrivateKey.matches("(?i:.*A00706052B8104000A.*)")) {
                algorithmEmployed = AlgorithmEmployed.SECP256K1;
            } else {
                throw new EOSFormatterError("DER format of private key is incorrect!");
            }
            switch (algorithmEmployed) {
                case SECP256R1: {
                    eosFormattedPrivateKey = eosFormattedPrivateKey.substring(PATTERN_STRING_PEM_PREFIX_PRIVATE_KEY_SECP256R1.length(), eosFormattedPrivateKey.length() - PATTERN_STRING_PEM_SUFFIX_PRIVATE_KEY_SECP256R1.length());
                    break;
                }
                case SECP256K1: {
                    eosFormattedPrivateKey = eosFormattedPrivateKey.substring(PATTERN_STRING_PEM_PREFIX_PRIVATE_KEY_SECP256K1.length(), eosFormattedPrivateKey.length() - PATTERN_STRING_PEM_SUFFIX_PRIVATE_KEY_SECP256K1.length());
                    break;
                }
                default: {
                    throw new EOSFormatterError("Unsupported algorithm!");
                }
            }
            try {
                eosFormattedPrivateKey = EOSFormatter.encodePrivateKey(Hex.decode((String)eosFormattedPrivateKey), algorithmEmployed);
            }
            catch (Base58ManipulationError e) {
                throw new EOSFormatterError(e);
            }
            builder = new StringBuilder(eosFormattedPrivateKey);
            switch (algorithmEmployed) {
                case SECP256K1: {
                    break;
                }
                case SECP256R1: {
                    builder.insert(0, PATTERN_STRING_EOS_PREFIX_PVT_R1);
                    break;
                }
            }
        } else {
            throw new EOSFormatterError("This is not a PEM formatted private key!");
        }
        eosFormattedPrivateKey = builder.toString();
        return eosFormattedPrivateKey;
    }

    @NotNull
    public static String convertEOSPrivateKeyToPEMFormat(@NotNull String privateKeyEOS) throws EOSFormatterError {
        byte[] base58DecodedPrivateKey;
        AlgorithmEmployed algorithmEmployed;
        String pemFormattedPrivateKey = privateKeyEOS;
        if (pemFormattedPrivateKey.toUpperCase().contains(PATTERN_STRING_EOS_PREFIX_PVT_R1.toUpperCase())) {
            algorithmEmployed = AlgorithmEmployed.SECP256R1;
            pemFormattedPrivateKey = pemFormattedPrivateKey.split(PATTERN_STRING_EOS_PREFIX_PVT_R1)[1];
        } else {
            algorithmEmployed = AlgorithmEmployed.SECP256K1;
        }
        try {
            base58DecodedPrivateKey = EOSFormatter.decodePrivateKey(pemFormattedPrivateKey, algorithmEmployed);
        }
        catch (Exception e) {
            throw new EOSFormatterError("An error occurred while Base58 decoding the EOS key!", e);
        }
        pemFormattedPrivateKey = Hex.toHexString((byte[])base58DecodedPrivateKey);
        switch (algorithmEmployed) {
            case SECP256R1: {
                pemFormattedPrivateKey = PATTERN_STRING_PEM_PREFIX_PRIVATE_KEY_SECP256R1 + pemFormattedPrivateKey + PATTERN_STRING_PEM_SUFFIX_PRIVATE_KEY_SECP256R1;
                break;
            }
            case SECP256K1: {
                pemFormattedPrivateKey = PATTERN_STRING_PEM_PREFIX_PRIVATE_KEY_SECP256K1 + pemFormattedPrivateKey + PATTERN_STRING_PEM_SUFFIX_PRIVATE_KEY_SECP256K1;
                break;
            }
            default: {
                throw new EOSFormatterError("Unsupported algorithm!");
            }
        }
        if (pemFormattedPrivateKey.length() <= 4) {
            throw new EOSFormatterError("The EOS private key provided is invalid!");
        }
        int i = (pemFormattedPrivateKey.length() - 4) / 2;
        String correctedLength = Integer.toHexString(i);
        pemFormattedPrivateKey = pemFormattedPrivateKey.substring(0, 2) + correctedLength + pemFormattedPrivateKey.substring(4);
        try {
            pemFormattedPrivateKey = EOSFormatter.derToPEM(Hex.decode((String)pemFormattedPrivateKey), PEMObjectType.PRIVATEKEY);
        }
        catch (DerToPemConversionError e) {
            throw new EOSFormatterError(e);
        }
        return pemFormattedPrivateKey;
    }

    public static String extractSerializedTransactionFromSignable(@NotNull String eosTransaction) throws EOSFormatterError {
        if (eosTransaction.isEmpty()) {
            throw new EOSFormatterError("Signable transaction can't be empty!");
        }
        if (eosTransaction.length() <= 129) {
            throw new EOSFormatterError(String.format("Length of the signable transaction must be larger than %s", 129));
        }
        try {
            return eosTransaction.substring(64, eosTransaction.length() - 64);
        }
        catch (Exception ex) {
            throw new EOSFormatterError("Something went wrong when trying to extract serialized transaction from signable transaction.", ex);
        }
    }

    public static String prepareSerializedTransactionForSigning(@NotNull String serializedTransaction, @NotNull String chainId, @NotNull String serializedContextFreeData) throws EOSFormatterError {
        if (serializedTransaction.isEmpty() || chainId.isEmpty() || serializedContextFreeData.isEmpty()) {
            throw new EOSFormatterError("Chain id, serialized transaction, and serialized context free data can't be empty!");
        }
        String signableTransaction = chainId + serializedTransaction + serializedContextFreeData;
        if (signableTransaction.length() <= 129) {
            throw new EOSFormatterError(String.format("Length of the signable transaction must be larger than %s", 129));
        }
        return signableTransaction;
    }

    public static String prepareSerializedTransactionForSigning(@NotNull String serializedTransaction, @NotNull String chainId) throws EOSFormatterError {
        return EOSFormatter.prepareSerializedTransactionForSigning(serializedTransaction, chainId, Hex.toHexString((byte[])new byte[32]));
    }

    @NotNull
    private static String derToPEM(@NotNull byte[] derEncodedByteArray, @NotNull PEMObjectType pemObjectType) throws DerToPemConversionError {
        StringBuilder pemForm;
        block8: {
            pemForm = new StringBuilder();
            try {
                if (pemObjectType.equals((Object)PEMObjectType.PRIVATEKEY)) {
                    pemForm.append(PEM_HEADER_PRIVATE_KEY);
                } else if (pemObjectType.equals((Object)PEMObjectType.PUBLICKEY)) {
                    pemForm.append(PEM_HEADER_PUBLIC_KEY);
                } else {
                    throw new DerToPemConversionError("Error converting DER encoded key to PEM format!");
                }
                pemForm.append("\n");
                String base64EncodedByteArray = new String(Base64.encode((byte[])derEncodedByteArray));
                pemForm.append(base64EncodedByteArray);
                pemForm.append("\n");
                if (pemObjectType.equals((Object)PEMObjectType.PRIVATEKEY)) {
                    pemForm.append(PEM_FOOTER_PRIVATE_KEY);
                    break block8;
                }
                if (pemObjectType.equals((Object)PEMObjectType.PUBLICKEY)) {
                    pemForm.append(PEM_FOOTER_PUBLIC_KEY);
                    break block8;
                }
                throw new DerToPemConversionError("Error converting DER encoded key to PEM format!");
            }
            catch (Exception e) {
                throw new DerToPemConversionError("Error converting DER encoded key to PEM format!", e);
            }
        }
        return pemForm.toString();
    }

    @NotNull
    private static byte[] decodePrivateKey(@NotNull String strKey, AlgorithmEmployed keyType) throws Base58ManipulationError {
        byte[] decodedKey;
        if (strKey.isEmpty()) {
            throw new IllegalArgumentException("Input key to decode can't be empty!");
        }
        try {
            byte[] base58Decoded = Base58.decode((String)strKey);
            byte[] firstCheckSum = Arrays.copyOfRange(base58Decoded, base58Decoded.length - 4, base58Decoded.length);
            decodedKey = Arrays.copyOfRange(base58Decoded, 0, base58Decoded.length - 4);
            switch (keyType) {
                case SECP256R1: {
                    byte[] secp256r1Suffix = SECP256R1_AND_PRIME256V1_CHECKSUM_VALIDATION_SUFFIX.getBytes();
                    if (!EOSFormatter.invalidRipeMD160CheckSum(decodedKey, firstCheckSum, secp256r1Suffix)) break;
                    throw new IllegalArgumentException("Input key has invalid checksum!");
                }
                case PRIME256V1: {
                    byte[] prime256v1Suffix = SECP256R1_AND_PRIME256V1_CHECKSUM_VALIDATION_SUFFIX.getBytes();
                    if (!EOSFormatter.invalidRipeMD160CheckSum(decodedKey, firstCheckSum, prime256v1Suffix)) break;
                    throw new IllegalArgumentException("Input key has invalid checksum!");
                }
                case SECP256K1: {
                    if (!EOSFormatter.invalidSha256x2CheckSum(decodedKey, firstCheckSum)) break;
                    throw new IllegalArgumentException("Input key has invalid checksum!");
                }
                default: {
                    throw new Base58ManipulationError("Unsupported algorithm!");
                }
            }
            if (decodedKey.length > 32 && keyType != AlgorithmEmployed.SECP256R1 && (decodedKey = Arrays.copyOfRange(decodedKey, 1, decodedKey.length)).length > 32 && decodedKey[32] == Integer.valueOf(1).byteValue()) {
                decodedKey = Arrays.copyOfRange(decodedKey, 0, decodedKey.length - 1);
            }
        }
        catch (Exception ex) {
            throw new Base58ManipulationError("An error occurred while Base58 decoding the EOS key!", ex);
        }
        return decodedKey;
    }

    @NotNull
    public static String encodePrivateKey(@NotNull byte[] pemKey, @NotNull AlgorithmEmployed keyType) throws Base58ManipulationError {
        byte[] checkSum;
        String base58Key = LEGACY_CHECKSUM_VALIDATION_SUFFIX;
        switch (keyType) {
            case SECP256R1: {
                checkSum = EOSFormatter.extractCheckSumRIPEMD160(pemKey, SECP256R1_AND_PRIME256V1_CHECKSUM_VALIDATION_SUFFIX.getBytes());
                break;
            }
            case PRIME256V1: {
                checkSum = EOSFormatter.extractCheckSumRIPEMD160(pemKey, SECP256R1_AND_PRIME256V1_CHECKSUM_VALIDATION_SUFFIX.getBytes());
                break;
            }
            case SECP256K1: {
                pemKey = Bytes.concat((byte[][])new byte[][]{{Integer.valueOf(128).byteValue()}, pemKey});
                checkSum = EOSFormatter.extractCheckSumSha256x2(pemKey);
                break;
            }
            default: {
                throw new Base58ManipulationError("Could not generate checksum!");
            }
        }
        base58Key = Base58.encode((byte[])Bytes.concat((byte[][])new byte[][]{pemKey, checkSum}));
        if (base58Key.isEmpty()) {
            throw new Base58ManipulationError("Unable to Base58 encode object!");
        }
        return base58Key;
    }

    @NotNull
    public static String encodePublicKey(@NotNull byte[] pemKey, @NotNull AlgorithmEmployed keyType, boolean isLegacy) throws Base58ManipulationError {
        String base58Key = LEGACY_CHECKSUM_VALIDATION_SUFFIX;
        if (pemKey.length == 0) {
            throw new IllegalArgumentException("Input key to decode can't be empty!");
        }
        try {
            byte[] checkSum;
            switch (keyType) {
                case SECP256K1: {
                    if (isLegacy) {
                        checkSum = EOSFormatter.extractCheckSumRIPEMD160(pemKey, LEGACY_CHECKSUM_VALIDATION_SUFFIX.getBytes());
                        break;
                    }
                    checkSum = EOSFormatter.extractCheckSumRIPEMD160(pemKey, SECP256K1_CHECKSUM_VALIDATION_SUFFIX.getBytes());
                    break;
                }
                case SECP256R1: {
                    checkSum = EOSFormatter.extractCheckSumRIPEMD160(pemKey, SECP256R1_AND_PRIME256V1_CHECKSUM_VALIDATION_SUFFIX.getBytes());
                    break;
                }
                default: {
                    throw new Base58ManipulationError("Unsupported algorithm!");
                }
            }
            base58Key = Base58.encode((byte[])Bytes.concat((byte[][])new byte[][]{pemKey, checkSum}));
            if (base58Key.equals(LEGACY_CHECKSUM_VALIDATION_SUFFIX)) {
                throw new Base58ManipulationError("Unable to Base58 encode object!");
            }
        }
        catch (Exception ex) {
            throw new Base58ManipulationError("Unable to Base58 encode object!", ex);
        }
        StringBuilder builder = new StringBuilder(base58Key);
        switch (keyType) {
            case SECP256K1: {
                if (isLegacy) {
                    builder.insert(0, PATTERN_STRING_EOS_PREFIX_EOS);
                    break;
                }
                builder.insert(0, PATTERN_STRING_EOS_PREFIX_PUB_K1);
                break;
            }
            case SECP256R1: {
                builder.insert(0, PATTERN_STRING_EOS_PREFIX_PUB_R1);
                break;
            }
        }
        base58Key = builder.toString();
        return base58Key;
    }

    @NotNull
    public static byte[] decodePublicKey(@NotNull String strKey, String keyPrefix) throws Base58ManipulationError {
        if (strKey.isEmpty()) {
            throw new IllegalArgumentException("Input key to decode can't be empty.");
        }
        byte[] decodedKey = null;
        try {
            byte[] base58Decoded = Base58.decode((String)strKey);
            byte[] firstCheckSum = Arrays.copyOfRange(base58Decoded, base58Decoded.length - 4, base58Decoded.length);
            decodedKey = Arrays.copyOfRange(base58Decoded, 0, base58Decoded.length - 4);
            switch (keyPrefix) {
                case "PUB_R1_": {
                    if (!EOSFormatter.invalidRipeMD160CheckSum(decodedKey, firstCheckSum, SECP256R1_AND_PRIME256V1_CHECKSUM_VALIDATION_SUFFIX.getBytes())) break;
                    throw new IllegalArgumentException("Input key has invalid checksum!");
                }
                case "PUB_K1_": {
                    if (!EOSFormatter.invalidRipeMD160CheckSum(decodedKey, firstCheckSum, SECP256K1_CHECKSUM_VALIDATION_SUFFIX.getBytes())) break;
                    throw new IllegalArgumentException("Input key has invalid checksum!");
                }
                case "EOS": {
                    if (!EOSFormatter.invalidRipeMD160CheckSum(decodedKey, firstCheckSum, LEGACY_CHECKSUM_VALIDATION_SUFFIX.getBytes())) break;
                    throw new IllegalArgumentException("Input key has invalid checksum!");
                }
            }
        }
        catch (Exception ex) {
            throw new Base58ManipulationError("An error occurred while Base58 decoding the EOS key!", ex);
        }
        return decodedKey;
    }

    private static boolean invalidRipeMD160CheckSum(@NotNull byte[] inputKey, @NotNull byte[] checkSumToValidate, @NotNull byte[] keyTypeByteArray) {
        if (inputKey.length == 0 || checkSumToValidate.length == 0) {
            throw new IllegalArgumentException("Input key, checksum and key type to validate can't be empty!");
        }
        byte[] keyWithType = Bytes.concat((byte[][])new byte[][]{inputKey, keyTypeByteArray});
        byte[] digestRIPEMD160 = EOSFormatter.digestRIPEMD160(keyWithType);
        byte[] checkSumFromInputKey = Arrays.copyOfRange(digestRIPEMD160, 0, 4);
        return !Arrays.equals(checkSumToValidate, checkSumFromInputKey);
    }

    private static boolean invalidSha256x2CheckSum(@NotNull byte[] inputKey, @NotNull byte[] checkSumToValidate) {
        if (inputKey.length == 0 || checkSumToValidate.length == 0) {
            throw new IllegalArgumentException("Input key, checksum and key type to validate can't be empty!");
        }
        byte[] sha256x2 = Sha256Hash.hashTwice((byte[])inputKey);
        byte[] checkSumFromInputKey = Arrays.copyOfRange(sha256x2, 0, 4);
        return !Arrays.equals(checkSumToValidate, checkSumFromInputKey);
    }

    @NotNull
    private static byte[] digestRIPEMD160(@NotNull byte[] input) {
        RIPEMD160Digest digest = new RIPEMD160Digest();
        byte[] output = new byte[digest.getDigestSize()];
        digest.update(input, 0, input.length);
        digest.doFinal(output, 0);
        return output;
    }

    @NotNull
    private static byte[] extractCheckSumRIPEMD160(@NotNull byte[] pemKey, byte[] keyTypeByteArray) {
        if (keyTypeByteArray != null) {
            pemKey = Bytes.concat((byte[][])new byte[][]{pemKey, keyTypeByteArray});
        }
        byte[] ripemd160Digest = EOSFormatter.digestRIPEMD160(pemKey);
        return Arrays.copyOfRange(ripemd160Digest, 0, 4);
    }

    @NotNull
    private static byte[] extractCheckSumSha256x2(@NotNull byte[] pemKey) {
        byte[] sha256x2 = Sha256Hash.hashTwice((byte[])pemKey);
        return Arrays.copyOfRange(sha256x2, 0, 4);
    }

    @NotNull
    private static byte[] decompressPublickey(byte[] compressedPublicKey, AlgorithmEmployed algorithmEmployed) throws EOSFormatterError {
        try {
            ECNamedCurveParameterSpec parameterSpec = ECNamedCurveTable.getParameterSpec((String)algorithmEmployed.getString());
            ECPoint ecPoint = parameterSpec.getCurve().decodePoint(compressedPublicKey);
            byte[] x = ecPoint.getXCoord().getEncoded();
            byte[] y = ecPoint.getYCoord().getEncoded();
            if (y.length > 32) {
                y = Arrays.copyOfRange(y, 1, y.length);
            }
            return Bytes.concat((byte[][])new byte[][]{{4}, x, y});
        }
        catch (Exception e) {
            throw new EOSFormatterError("Problem decompressing public key!", e);
        }
    }

    @NotNull
    private static byte[] compressPublickey(byte[] compressedPublicKey, AlgorithmEmployed algorithmEmployed) throws EOSFormatterError {
        try {
            ECNamedCurveParameterSpec parameterSpec = ECNamedCurveTable.getParameterSpec((String)algorithmEmployed.getString());
            ECPoint ecPoint = parameterSpec.getCurve().decodePoint(compressedPublicKey);
            byte[] x = ecPoint.getXCoord().getEncoded();
            byte[] y = ecPoint.getYCoord().getEncoded();
            BigInteger bigIntegerY = new BigInteger(Hex.toHexString((byte[])y), 16);
            BigInteger bigIntegerTwo = BigInteger.valueOf(2L);
            BigInteger remainder = bigIntegerY.mod(bigIntegerTwo);
            byte compressionPrefix = remainder.equals(BigInteger.ZERO) ? (byte)2 : 3;
            return Bytes.concat((byte[][])new byte[][]{{compressionPrefix}, x});
        }
        catch (Exception e) {
            throw new EOSFormatterError("Problem compressing public key!", e);
        }
    }

    private static BigInteger checkAndHandleLowS(BigInteger s, AlgorithmEmployed keyType) throws LowSVerificationError {
        if (!EOSFormatter.isLowS(s, keyType)) {
            switch (keyType) {
                case SECP256R1: {
                    return CURVE_R1.getN().subtract(s);
                }
            }
            return CURVE_K1.getN().subtract(s);
        }
        return s;
    }

    private static boolean isLowS(BigInteger s, AlgorithmEmployed keyType) throws LowSVerificationError {
        int compareResult;
        switch (keyType) {
            case SECP256R1: {
                compareResult = s.compareTo(HALF_CURVE_ORDER_R1);
                break;
            }
            case SECP256K1: {
                compareResult = s.compareTo(HALF_CURVE_ORDER_K1);
                break;
            }
            default: {
                throw new LowSVerificationError("Unsupported algorithm!");
            }
        }
        return compareResult == 0 || compareResult == -1;
    }

    private static byte[] addCheckSumToSignature(byte[] signature, byte[] keyTypeByteArray) {
        byte[] signatureWithKeyType = Bytes.concat((byte[][])new byte[][]{signature, keyTypeByteArray});
        byte[] signatureRipemd160 = EOSFormatter.digestRIPEMD160(signatureWithKeyType);
        byte[] checkSum = Arrays.copyOfRange(signatureRipemd160, 0, 4);
        return Bytes.concat((byte[][])new byte[][]{signature, checkSum});
    }

    private static boolean isCanonical(byte[] signature) {
        return !((signature[1] & Integer.valueOf(128).byteValue()) != Integer.valueOf(0).byteValue() || signature[1] == Integer.valueOf(0).byteValue() && (signature[2] & Integer.valueOf(128).byteValue()) == Integer.valueOf(0).byteValue() || (signature[33] & Integer.valueOf(128).byteValue()) != Integer.valueOf(0).byteValue() || signature[33] == Integer.valueOf(0).byteValue() && (signature[34] & Integer.valueOf(128).byteValue()) == Integer.valueOf(0).byteValue());
    }

    private static int getRecoveryId(BigInteger r, BigInteger s, Sha256Hash sha256HashMessage, byte[] publicKey, AlgorithmEmployed keyType) {
        for (int i = 0; i < 4; ++i) {
            byte[] recoveredPublicKey = EOSFormatter.recoverPublicKeyFromSignature(i, r, s, sha256HashMessage, true, keyType);
            if (!Arrays.equals(publicKey, recoveredPublicKey)) continue;
            return i;
        }
        return -1;
    }

    private static byte[] recoverPublicKeyFromSignature(int recId, BigInteger r, BigInteger s, @NotNull Sha256Hash message, boolean compressed, AlgorithmEmployed keyType) {
        ECCurve.Fp curve;
        ECPoint g;
        BigInteger n;
        Preconditions.checkArgument((recId >= 0 ? 1 : 0) != 0, (Object)"recId must be positive");
        Preconditions.checkArgument((r.signum() >= 0 ? 1 : 0) != 0, (Object)"r must be positive");
        Preconditions.checkArgument((s.signum() >= 0 ? 1 : 0) != 0, (Object)"s must be positive");
        switch (keyType) {
            case SECP256R1: {
                n = ecParamsR1.getN();
                g = ecParamsR1.getG();
                curve = (ECCurve.Fp)ecParamsR1.getCurve();
                break;
            }
            default: {
                n = ecParamsK1.getN();
                g = ecParamsK1.getG();
                curve = (ECCurve.Fp)ecParamsK1.getCurve();
            }
        }
        BigInteger i = BigInteger.valueOf((long)recId / 2L);
        BigInteger x = r.add(i.multiply(n));
        BigInteger prime = curve.getQ();
        if (x.compareTo(prime) >= 0) {
            return null;
        }
        ECPoint R = EOSFormatter.decompressKey(x, (recId & 1) == 1, keyType);
        if (!R.multiply(n).isInfinity()) {
            return null;
        }
        BigInteger e = message.toBigInteger();
        BigInteger eInv = BigInteger.ZERO.subtract(e).mod(n);
        BigInteger rInv = r.modInverse(n);
        BigInteger srInv = rInv.multiply(s).mod(n);
        BigInteger eInvrInv = rInv.multiply(eInv).mod(n);
        ECPoint q = ECAlgorithms.sumOfTwoMultiplies((ECPoint)g, (BigInteger)eInvrInv, (ECPoint)R, (BigInteger)srInv);
        return q.getEncoded(compressed);
    }

    private static ECPoint decompressKey(BigInteger xBN, boolean yBit, AlgorithmEmployed keyType) {
        ECCurve.Fp curve;
        switch (keyType) {
            case SECP256R1: {
                curve = (ECCurve.Fp)ecParamsR1.getCurve();
                break;
            }
            default: {
                curve = (ECCurve.Fp)ecParamsK1.getCurve();
            }
        }
        X9IntegerConverter x9 = new X9IntegerConverter();
        byte[] compEnc = x9.integerToBytes(xBN, 1 + x9.getByteLength((ECCurve)curve));
        compEnc[0] = yBit ? 3 : 2;
        return curve.decodePoint(compEnc);
    }

    static {
        CURVE_PARAMS_R1 = CustomNamedCurves.getByName((String)SECP256_R1);
        CURVE_PARAMS_K1 = CustomNamedCurves.getByName((String)SECP256_K1);
        X9ECParameters paramsR1 = SECNamedCurves.getByName((String)SECP256_R1);
        ecParamsR1 = new ECDomainParameters(paramsR1.getCurve(), paramsR1.getG(), paramsR1.getN(), paramsR1.getH());
        X9ECParameters paramsK1 = SECNamedCurves.getByName((String)SECP256_K1);
        ecParamsK1 = new ECDomainParameters(paramsK1.getCurve(), paramsK1.getG(), paramsK1.getN(), paramsK1.getH());
        FixedPointUtil.precompute((ECPoint)CURVE_PARAMS_R1.getG());
        CURVE_R1 = new ECDomainParameters(CURVE_PARAMS_R1.getCurve(), CURVE_PARAMS_R1.getG(), CURVE_PARAMS_R1.getN(), CURVE_PARAMS_R1.getH());
        HALF_CURVE_ORDER_R1 = CURVE_PARAMS_R1.getN().shiftRight(1);
        CURVE_K1 = new ECDomainParameters(CURVE_PARAMS_K1.getCurve(), CURVE_PARAMS_K1.getG(), CURVE_PARAMS_K1.getN(), CURVE_PARAMS_K1.getH());
        HALF_CURVE_ORDER_K1 = CURVE_PARAMS_K1.getN().shiftRight(1);
    }

    private static enum PEMObjectType {
        PUBLICKEY("PUBLIC KEY"),
        PRIVATEKEY("PRIVATE KEY"),
        SIGNATURE("SIGNATURE");

        private String value;

        private PEMObjectType(String value) {
            this.value = value;
        }

        public String getString() {
            return this.value;
        }
    }
}

