package io.neow3j.compiler;

// Taken from com.sun.tools.classfile.Opcode

/**
 * Contains all opcodes from the JVM instruction set as documented in the Java Virtual Machine
 * Specification Java SE 8 Edition, which can be found
 * <a href="https://docs.oracle.com/javase/specs/jvms/se8/jvms8.pdf">here</a> in chapter 6.
 */
public enum JVMOpcode {

    NOP(0),
    ACONST_NULL(1),
    ICONST_M1(2),
    ICONST_0(3),
    ICONST_1(4),
    ICONST_2(5),
    ICONST_3(6),
    ICONST_4(7),
    ICONST_5(8),
    LCONST_0(9),
    LCONST_1(10),
    FCONST_0(11),
    FCONST_1(12),
    FCONST_2(13),
    DCONST_0(14),
    DCONST_1(15),
    BIPUSH(16, Kind.BYTE),
    SIPUSH(17, Kind.SHORT),
    LDC(18, Kind.CPREF),
    LDC_W(19, Kind.CPREF_W),
    LDC2_W(20, Kind.CPREF_W),
    ILOAD(21, Kind.LOCAL),
    LLOAD(22, Kind.LOCAL),
    FLOAD(23, Kind.LOCAL),
    DLOAD(24, Kind.LOCAL),
    ALOAD(25, Kind.LOCAL),
    ILOAD_0(26),
    ILOAD_1(27),
    ILOAD_2(28),
    ILOAD_3(29),
    LLOAD_0(30),
    LLOAD_1(31),
    LLOAD_2(32),
    LLOAD_3(33),
    FLOAD_0(34),
    FLOAD_1(35),
    FLOAD_2(36),
    FLOAD_3(37),
    DLOAD_0(38),
    DLOAD_1(39),
    DLOAD_2(40),
    DLOAD_3(41),
    ALOAD_0(42),
    ALOAD_1(43),
    ALOAD_2(44),
    ALOAD_3(45),
    IALOAD(46),
    LALOAD(47),
    FALOAD(48),
    DALOAD(49),
    AALOAD(50),
    BALOAD(51),
    CALOAD(52),
    SALOAD(53),
    ISTORE(54, Kind.LOCAL),
    LSTORE(55, Kind.LOCAL),
    FSTORE(56, Kind.LOCAL),
    DSTORE(57, Kind.LOCAL),
    ASTORE(58, Kind.LOCAL),
    ISTORE_0(59),
    ISTORE_1(60),
    ISTORE_2(61),
    ISTORE_3(62),
    LSTORE_0(63),
    LSTORE_1(64),
    LSTORE_2(65),
    LSTORE_3(66),
    FSTORE_0(67),
    FSTORE_1(68),
    FSTORE_2(69),
    FSTORE_3(70),
    DSTORE_0(71),
    DSTORE_1(72),
    DSTORE_2(73),
    DSTORE_3(74),
    ASTORE_0(75),
    ASTORE_1(76),
    ASTORE_2(77),
    ASTORE_3(78),
    IASTORE(79),
    LASTORE(80),
    FASTORE(81),
    DASTORE(82),
    AASTORE(83),
    BASTORE(84),
    CASTORE(85),
    SASTORE(86),
    POP(87),
    POP2(88),
    DUP(89),
    DUP_X1(90),
    DUP_X2(91),
    DUP2(92),
    DUP2_X1(93),
    DUP2_X2(94),
    SWAP(95),
    IADD(96),
    LADD(97),
    FADD(98),
    DADD(99),
    ISUB(100),
    LSUB(101),
    FSUB(102),
    DSUB(103),
    IMUL(104),
    LMUL(105),
    FMUL(106),
    DMUL(107),
    IDIV(108),
    LDIV(109),
    FDIV(110),
    DDIV(111),
    IREM(112),
    LREM(113),
    FREM(114),
    DREM(115),
    INEG(116),
    LNEG(117),
    FNEG(118),
    DNEG(119),
    ISHL(120),
    LSHL(121),
    ISHR(122),
    LSHR(123),
    IUSHR(124),
    LUSHR(125),
    IAND(126),
    LAND(127),
    IOR(128),
    LOR(129),
    IXOR(130),
    LXOR(131),
    IINC(132, Kind.LOCAL_BYTE),
    I2L(133),
    I2F(134),
    I2D(135),
    L2I(136),
    L2F(137),
    L2D(138),
    F2I(139),
    F2L(140),
    F2D(141),
    D2I(142),
    D2L(143),
    D2F(144),
    I2B(145),
    I2C(146),
    I2S(147),
    LCMP(148),
    FCMPL(149),
    FCMPG(150),
    DCMPL(151),
    DCMPG(152),
    IFEQ(153, Kind.BRANCH),
    IFNE(154, Kind.BRANCH),
    IFLT(155, Kind.BRANCH),
    IFGE(156, Kind.BRANCH),
    IFGT(157, Kind.BRANCH),
    IFLE(158, Kind.BRANCH),
    IF_ICMPEQ(159, Kind.BRANCH),
    IF_ICMPNE(160, Kind.BRANCH),
    IF_ICMPLT(161, Kind.BRANCH),
    IF_ICMPGE(162, Kind.BRANCH),
    IF_ICMPGT(163, Kind.BRANCH),
    IF_ICMPLE(164, Kind.BRANCH),
    IF_ACMPEQ(165, Kind.BRANCH),
    IF_ACMPNE(166, Kind.BRANCH),
    GOTO(167, Kind.BRANCH),
    JSR(168, Kind.BRANCH),
    RET(169, Kind.LOCAL),
    TABLESWITCH(170, Kind.DYNAMIC),
    LOOKUPSWITCH(171, Kind.DYNAMIC),
    IRETURN(172),
    LRETURN(173),
    FRETURN(174),
    DRETURN(175),
    ARETURN(176),
    RETURN(177),
    GETSTATIC(178, Kind.CPREF_W),
    PUTSTATIC(179, Kind.CPREF_W),
    GETFIELD(180, Kind.CPREF_W),
    PUTFIELD(181, Kind.CPREF_W),
    INVOKEVIRTUAL(182, Kind.CPREF_W),
    INVOKESPECIAL(183, Kind.CPREF_W),
    INVOKESTATIC(184, Kind.CPREF_W),
    INVOKEINTERFACE(185, Kind.CPREF_W_UBYTE_ZERO),
    INVOKEDYNAMIC(186, Kind.CPREF_W_UBYTE_ZERO),
    NEW(187, Kind.CPREF_W),
    NEWARRAY(188, Kind.ATYPE),
    ANEWARRAY(189, Kind.CPREF_W),
    ARRAYLENGTH(190),
    ATHROW(191),
    CHECKCAST(192, Kind.CPREF_W),
    INSTANCEOF(193, Kind.CPREF_W),
    MONITORENTER(194),
    MONITOREXIT(195),
    WIDE(196),
    MULTIANEWARRAY(197, Kind.CPREF_W_UBYTE),
    IFNULL(198, Kind.BRANCH),
    IFNONNULL(199, Kind.BRANCH),
    GOTO_W(200, Kind.BRANCH_W),
    JSR_W(201, Kind.BRANCH_W);

    private int opcode;
    private Kind kind;
    private static JVMOpcode[] opcodes = values();

    JVMOpcode(int opcode) {
        this(opcode, Kind.NO_OPERANDS);
    }

    JVMOpcode(int opcode, Kind kind) {
        this.opcode = opcode;
        this.kind = kind;
    }

    public static JVMOpcode get(int opcode) {
        if (opcode < 0 || opcode > 201) {
            return null;
        }
        return opcodes[opcode];
    }

    public int getOpcode() {
        return this.opcode;
    }

    public Kind getKind() {
        return this.kind;
    }

    public enum Kind {
        NO_OPERANDS(1),
        ATYPE(2),
        BRANCH(3),
        BRANCH_W(5),
        BYTE(2),
        CPREF(2),
        CPREF_W(3),
        CPREF_W_UBYTE(4),
        CPREF_W_UBYTE_ZERO(5),
        DYNAMIC(-1),
        LOCAL(2),
        LOCAL_BYTE(3),
        SHORT(3),
        WIDE_NO_OPERANDS(2),
        WIDE_LOCAL(4),
        WIDE_CPREF_W(4),
        WIDE_CPREF_W_SHORT(6),
        WIDE_LOCAL_SHORT(6),
        UNKNOWN(1);

        public final int length;

        Kind(int kindLenght) {
            this.length = kindLenght;
        }
    }
}
