/*
 * Copyright 2020 NEM
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.nem.symbol.sdk.model.transaction;

import io.nem.symbol.core.utils.ConvertUtils;
import io.nem.symbol.sdk.model.mosaic.MosaicSupplyChangeActionType;
import io.nem.symbol.sdk.model.mosaic.UnresolvedMosaicId;
import io.nem.symbol.sdk.model.network.NetworkType;
import java.math.BigInteger;
import org.apache.commons.lang3.Validate;

/** Factory of {@link MosaicSupplyChangeTransaction} */
public class MosaicSupplyChangeTransactionFactory
    extends TransactionFactory<MosaicSupplyChangeTransaction> {

  private final UnresolvedMosaicId mosaicId;
  private final MosaicSupplyChangeActionType action;
  private final BigInteger delta;

  private MosaicSupplyChangeTransactionFactory(
      NetworkType networkType,
      Deadline deadline,
      UnresolvedMosaicId mosaicId,
      MosaicSupplyChangeActionType action,
      BigInteger delta) {
    super(TransactionType.MOSAIC_SUPPLY_CHANGE, networkType, deadline);
    Validate.notNull(mosaicId, "UnresolvedMosaicId must not be null");
    Validate.notNull(action, "Action must not be null");
    Validate.notNull(delta, "Delta must not be null");
    ConvertUtils.validateNotNegative(delta);
    this.mosaicId = mosaicId;
    this.action = action;
    this.delta = delta;
  }

  /**
   * Static create method for factory.
   *
   * @param networkType Network type.
   * @param deadline the deadline
   * @param mosaicId Mosaic id.
   * @param action Action.
   * @param delta Delta.
   * @return Mosaic supply change transaction.
   */
  public static MosaicSupplyChangeTransactionFactory create(
      NetworkType networkType,
      Deadline deadline,
      UnresolvedMosaicId mosaicId,
      MosaicSupplyChangeActionType action,
      BigInteger delta) {
    return new MosaicSupplyChangeTransactionFactory(networkType, deadline, mosaicId, action, delta);
  }

  /**
   * Returns mosaic id.
   *
   * @return BigInteger
   */
  public UnresolvedMosaicId getMosaicId() {
    return mosaicId;
  }

  /**
   * Returns mosaic supply type.
   *
   * @return {@link MosaicSupplyChangeActionType}
   */
  public MosaicSupplyChangeActionType getAction() {
    return action;
  }

  /**
   * Returns amount of mosaics added or removed.
   *
   * @return BigInteger
   */
  public BigInteger getDelta() {
    return delta;
  }

  @Override
  public MosaicSupplyChangeTransaction build() {
    return new MosaicSupplyChangeTransaction(this);
  }
}
