/*
 * Decompiled with CFR 0.152.
 */
package io.aeron.driver.media;

import io.aeron.ChannelUri;
import io.aeron.ErrorCode;
import io.aeron.driver.Configuration;
import io.aeron.driver.exceptions.InvalidChannelException;
import io.aeron.driver.media.InterfaceSearchAddress;
import io.aeron.driver.media.NetworkUtil;
import io.aeron.driver.media.SocketAddressUtil;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.ProtocolFamily;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicInteger;
import org.agrona.BitUtil;

public final class UdpChannel {
    private static final byte[] HEX_TABLE = new byte[]{48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 97, 98, 99, 100, 101, 102};
    private static final AtomicInteger UNIQUE_CANONICAL_FORM_VALUE = new AtomicInteger();
    private final boolean hasExplicitControl;
    private final boolean isMulticast;
    private final boolean hasTag;
    private final long tag;
    private final int multicastTtl;
    private final InetSocketAddress remoteData;
    private final InetSocketAddress localData;
    private final InetSocketAddress remoteControl;
    private final InetSocketAddress localControl;
    private final String uriStr;
    private final String canonicalForm;
    private final NetworkInterface localInterface;
    private final ProtocolFamily protocolFamily;
    private final ChannelUri channelUri;

    private UdpChannel(Context context) {
        this.hasExplicitControl = context.hasExplicitControl;
        this.isMulticast = context.isMulticast;
        this.hasTag = context.hasTagId;
        this.tag = context.tagId;
        this.multicastTtl = context.multicastTtl;
        this.remoteData = context.remoteData;
        this.localData = context.localData;
        this.remoteControl = context.remoteControl;
        this.localControl = context.localControl;
        this.uriStr = context.uriStr;
        this.canonicalForm = context.canonicalForm;
        this.localInterface = context.localInterface;
        this.protocolFamily = context.protocolFamily;
        this.channelUri = context.channelUri;
    }

    public static UdpChannel parse(String channelUriString) {
        try {
            boolean hasNoDistinguishingCharacteristic;
            ChannelUri channelUri = ChannelUri.parse(channelUriString);
            UdpChannel.validateConfiguration(channelUri);
            InetSocketAddress endpointAddress = UdpChannel.getEndpointAddress(channelUri);
            InetSocketAddress explicitControlAddress = UdpChannel.getExplicitControlAddress(channelUri);
            String tagIdStr = channelUri.channelTag();
            String controlMode = channelUri.get("control-mode");
            boolean bl = hasNoDistinguishingCharacteristic = null == endpointAddress && null == explicitControlAddress && null == tagIdStr;
            if (hasNoDistinguishingCharacteristic && null == controlMode) {
                throw new IllegalArgumentException("Aeron URIs for UDP must specify an endpoint address, control address, tag-id, or control-mode");
            }
            if (null != endpointAddress && endpointAddress.isUnresolved()) {
                throw new UnknownHostException("could not resolve endpoint address: " + endpointAddress);
            }
            if (null != explicitControlAddress && explicitControlAddress.isUnresolved()) {
                throw new UnknownHostException("could not resolve control address: " + explicitControlAddress);
            }
            Context context = new Context().uriStr(channelUriString).channelUri(channelUri).hasNoDistinguishingCharacteristic(hasNoDistinguishingCharacteristic);
            if (null != tagIdStr) {
                context.hasTagId(true).tagId(Long.parseLong(tagIdStr));
            }
            if (null == endpointAddress) {
                endpointAddress = new InetSocketAddress("0.0.0.0", 0);
            }
            if (endpointAddress.getAddress().isMulticastAddress()) {
                InetSocketAddress controlAddress = UdpChannel.getMulticastControlAddress(endpointAddress);
                InterfaceSearchAddress searchAddress = UdpChannel.getInterfaceSearchAddress(channelUri);
                NetworkInterface localInterface = UdpChannel.findInterface(searchAddress);
                InetSocketAddress resolvedAddress = UdpChannel.resolveToAddressOfInterface(localInterface, searchAddress);
                context.isMulticast(true).localControlAddress(resolvedAddress).remoteControlAddress(controlAddress).localDataAddress(resolvedAddress).remoteDataAddress(endpointAddress).localInterface(localInterface).multicastTtl(UdpChannel.getMulticastTtl(channelUri)).protocolFamily(NetworkUtil.getProtocolFamily(endpointAddress.getAddress())).canonicalForm(UdpChannel.canonicalise(resolvedAddress, endpointAddress));
            } else if (null != explicitControlAddress) {
                context.hasExplicitControl(true).remoteControlAddress(endpointAddress).remoteDataAddress(endpointAddress).localControlAddress(explicitControlAddress).localDataAddress(explicitControlAddress).protocolFamily(NetworkUtil.getProtocolFamily(endpointAddress.getAddress())).canonicalForm(UdpChannel.canonicalise(explicitControlAddress, endpointAddress));
            } else {
                InterfaceSearchAddress searchAddress = UdpChannel.getInterfaceSearchAddress(channelUri);
                InetSocketAddress localAddress = searchAddress.getInetAddress().isAnyLocalAddress() ? searchAddress.getAddress() : UdpChannel.resolveToAddressOfInterface(UdpChannel.findInterface(searchAddress), searchAddress);
                String uniqueCanonicalFormSuffix = hasNoDistinguishingCharacteristic ? "-" + UNIQUE_CANONICAL_FORM_VALUE.getAndAdd(1) : "";
                context.remoteControlAddress(endpointAddress).remoteDataAddress(endpointAddress).localControlAddress(localAddress).localDataAddress(localAddress).protocolFamily(NetworkUtil.getProtocolFamily(endpointAddress.getAddress())).canonicalForm(UdpChannel.canonicalise(localAddress, endpointAddress) + uniqueCanonicalFormSuffix);
            }
            return new UdpChannel(context);
        }
        catch (Exception ex) {
            throw new InvalidChannelException(ErrorCode.INVALID_CHANNEL, ex);
        }
    }

    private static InetSocketAddress getMulticastControlAddress(InetSocketAddress endpointAddress) throws UnknownHostException {
        byte[] addressAsBytes = endpointAddress.getAddress().getAddress();
        UdpChannel.validateDataAddress(addressAsBytes);
        int n = addressAsBytes.length - 1;
        addressAsBytes[n] = (byte)(addressAsBytes[n] + 1);
        return new InetSocketAddress(InetAddress.getByAddress(addressAsBytes), endpointAddress.getPort());
    }

    public static String canonicalise(InetSocketAddress localData, InetSocketAddress remoteData) {
        StringBuilder builder = new StringBuilder(48);
        builder.append("UDP-");
        UdpChannel.toHex(builder, localData.getAddress().getAddress()).append('-').append(localData.getPort());
        builder.append('-');
        UdpChannel.toHex(builder, remoteData.getAddress().getAddress()).append('-').append(remoteData.getPort());
        return builder.toString();
    }

    public InetSocketAddress remoteData() {
        return this.remoteData;
    }

    public InetSocketAddress localData() {
        return this.localData;
    }

    public InetSocketAddress remoteControl() {
        return this.remoteControl;
    }

    public InetSocketAddress localControl() {
        return this.localControl;
    }

    public ChannelUri channelUri() {
        return this.channelUri;
    }

    public int multicastTtl() {
        return this.multicastTtl;
    }

    public String canonicalForm() {
        return this.canonicalForm;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        UdpChannel that = (UdpChannel)o;
        return Objects.equals(this.canonicalForm, that.canonicalForm);
    }

    public int hashCode() {
        return this.canonicalForm != null ? this.canonicalForm.hashCode() : 0;
    }

    public String toString() {
        return this.canonicalForm;
    }

    public boolean isMulticast() {
        return this.isMulticast;
    }

    public NetworkInterface localInterface() {
        return this.localInterface;
    }

    public String originalUriString() {
        return this.uriStr;
    }

    public ProtocolFamily protocolFamily() {
        return this.protocolFamily;
    }

    public long tag() {
        return this.tag;
    }

    public boolean hasExplicitControl() {
        return this.hasExplicitControl;
    }

    public boolean hasTag() {
        return this.hasTag;
    }

    public boolean doesTagMatch(UdpChannel udpChannel) {
        if (!this.hasTag || !udpChannel.hasTag() || this.tag != udpChannel.tag()) {
            return false;
        }
        if (udpChannel.remoteData().getAddress().isAnyLocalAddress() && udpChannel.remoteData().getPort() == 0 && udpChannel.localData().getAddress().isAnyLocalAddress() && udpChannel.localData().getPort() == 0) {
            return true;
        }
        throw new IllegalArgumentException("matching tag has set endpoint or control address");
    }

    public static InetSocketAddress destinationAddress(ChannelUri uri) {
        try {
            UdpChannel.validateConfiguration(uri);
            return UdpChannel.getEndpointAddress(uri);
        }
        catch (Exception ex) {
            throw new InvalidChannelException(ErrorCode.INVALID_CHANNEL, ex);
        }
    }

    public String description() {
        StringBuilder builder = new StringBuilder("UdpChannel - ");
        if (null != this.localInterface) {
            builder.append("interface: ").append(this.localInterface.getDisplayName()).append(", ");
        }
        builder.append("localData: ").append(this.localData).append(", remoteData: ").append(this.remoteData).append(", ttl: ").append(this.multicastTtl);
        return builder.toString();
    }

    private static InterfaceSearchAddress getInterfaceSearchAddress(ChannelUri uri) throws UnknownHostException {
        String interfaceValue = uri.get("interface");
        if (null != interfaceValue) {
            return InterfaceSearchAddress.parse(interfaceValue);
        }
        return InterfaceSearchAddress.wildcard();
    }

    private static InetSocketAddress getEndpointAddress(ChannelUri uri) {
        String endpointValue = uri.get("endpoint");
        if (null != endpointValue) {
            return SocketAddressUtil.parse(endpointValue);
        }
        return null;
    }

    private static int getMulticastTtl(ChannelUri uri) {
        String ttlValue = uri.get("ttl");
        if (null != ttlValue) {
            return Integer.parseInt(ttlValue);
        }
        return Configuration.SOCKET_MULTICAST_TTL;
    }

    private static InetSocketAddress getExplicitControlAddress(ChannelUri uri) {
        String controlValue = uri.get("control");
        if (null != controlValue) {
            return SocketAddressUtil.parse(controlValue);
        }
        return null;
    }

    private static void validateDataAddress(byte[] addressAsBytes) {
        if (BitUtil.isEven(addressAsBytes[addressAsBytes.length - 1])) {
            throw new IllegalArgumentException("Multicast data address must be odd");
        }
    }

    private static void validateConfiguration(ChannelUri uri) {
        UdpChannel.validateMedia(uri);
    }

    private static void validateMedia(ChannelUri uri) {
        if (!"udp".equals(uri.media())) {
            throw new IllegalArgumentException("UdpChannel only supports UDP media: " + uri);
        }
    }

    private static InetSocketAddress resolveToAddressOfInterface(NetworkInterface localInterface, InterfaceSearchAddress searchAddress) {
        InetAddress interfaceAddress = NetworkUtil.findAddressOnInterface(localInterface, searchAddress.getInetAddress(), searchAddress.getSubnetPrefix());
        if (null == interfaceAddress) {
            throw new IllegalStateException();
        }
        return new InetSocketAddress(interfaceAddress, searchAddress.getPort());
    }

    private static NetworkInterface findInterface(InterfaceSearchAddress searchAddress) throws SocketException {
        NetworkInterface[] filteredInterfaces;
        for (NetworkInterface networkInterface : filteredInterfaces = NetworkUtil.filterBySubnet(searchAddress.getInetAddress(), searchAddress.getSubnetPrefix())) {
            if (!networkInterface.supportsMulticast() && !networkInterface.isLoopback()) continue;
            return networkInterface;
        }
        throw new IllegalArgumentException(UdpChannel.errorNoMatchingInterfaces(filteredInterfaces, searchAddress));
    }

    private static String errorNoMatchingInterfaces(NetworkInterface[] filteredInterfaces, InterfaceSearchAddress address) throws SocketException {
        StringBuilder builder = new StringBuilder().append("Unable to find multicast interface matching criteria: ").append(address.getAddress()).append('/').append(address.getSubnetPrefix());
        if (filteredInterfaces.length > 0) {
            builder.append(System.lineSeparator()).append("  Candidates:");
            for (NetworkInterface ifc : filteredInterfaces) {
                builder.append(System.lineSeparator()).append("  - Name: ").append(ifc.getDisplayName()).append(", addresses: ").append(ifc.getInterfaceAddresses()).append(", multicast: ").append(ifc.supportsMulticast());
            }
        }
        return builder.toString();
    }

    private static StringBuilder toHex(StringBuilder builder, byte[] bytes) {
        for (byte b : bytes) {
            builder.append((char)HEX_TABLE[b >> 4 & 0xF]);
            builder.append((char)HEX_TABLE[b & 0xF]);
        }
        return builder;
    }

    static class Context {
        long tagId;
        int multicastTtl;
        InetSocketAddress remoteData;
        InetSocketAddress localData;
        InetSocketAddress remoteControl;
        InetSocketAddress localControl;
        String uriStr;
        String canonicalForm;
        NetworkInterface localInterface;
        ProtocolFamily protocolFamily;
        ChannelUri channelUri;
        boolean hasExplicitControl = false;
        boolean isMulticast = false;
        boolean hasTagId = false;
        boolean hasNoDistinguishingCharacteristic = false;

        Context() {
        }

        Context uriStr(String uri) {
            this.uriStr = uri;
            return this;
        }

        Context remoteDataAddress(InetSocketAddress remoteData) {
            this.remoteData = remoteData;
            return this;
        }

        Context localDataAddress(InetSocketAddress localData) {
            this.localData = localData;
            return this;
        }

        Context remoteControlAddress(InetSocketAddress remoteControl) {
            this.remoteControl = remoteControl;
            return this;
        }

        Context localControlAddress(InetSocketAddress localControl) {
            this.localControl = localControl;
            return this;
        }

        Context canonicalForm(String canonicalForm) {
            this.canonicalForm = canonicalForm;
            return this;
        }

        Context localInterface(NetworkInterface networkInterface) {
            this.localInterface = networkInterface;
            return this;
        }

        Context protocolFamily(ProtocolFamily protocolFamily) {
            this.protocolFamily = protocolFamily;
            return this;
        }

        Context multicastTtl(int multicastTtl) {
            this.multicastTtl = multicastTtl;
            return this;
        }

        Context tagId(long tagId) {
            this.tagId = tagId;
            return this;
        }

        Context channelUri(ChannelUri channelUri) {
            this.channelUri = channelUri;
            return this;
        }

        Context hasExplicitControl(boolean hasExplicitControl) {
            this.hasExplicitControl = hasExplicitControl;
            return this;
        }

        Context isMulticast(boolean isMulticast) {
            this.isMulticast = isMulticast;
            return this;
        }

        Context hasTagId(boolean hasTagId) {
            this.hasTagId = hasTagId;
            return this;
        }

        Context hasNoDistinguishingCharacteristic(boolean hasNoDistinguishingCharacteristic) {
            this.hasNoDistinguishingCharacteristic = hasNoDistinguishingCharacteristic;
            return this;
        }
    }
}

