/*
 * Decompiled with CFR 0.152.
 */
package io.aeron.archive;

import io.aeron.archive.Archive;
import io.aeron.archive.RecordingSummary;
import io.aeron.archive.client.ArchiveException;
import io.aeron.archive.codecs.CatalogHeaderDecoder;
import io.aeron.archive.codecs.CatalogHeaderEncoder;
import io.aeron.archive.codecs.RecordingDescriptorDecoder;
import io.aeron.archive.codecs.RecordingDescriptorEncoder;
import io.aeron.archive.codecs.RecordingDescriptorHeaderDecoder;
import io.aeron.archive.codecs.RecordingDescriptorHeaderEncoder;
import java.io.File;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.StandardOpenOption;
import org.agrona.AsciiEncoding;
import org.agrona.BitUtil;
import org.agrona.CloseHelper;
import org.agrona.DirectBuffer;
import org.agrona.IoUtil;
import org.agrona.LangUtil;
import org.agrona.collections.ArrayUtil;
import org.agrona.concurrent.EpochClock;
import org.agrona.concurrent.UnsafeBuffer;

class Catalog
implements AutoCloseable {
    static final int PAGE_SIZE = 4096;
    static final int NULL_RECORD_ID = -1;
    static final int DESCRIPTOR_HEADER_LENGTH = 32;
    static final int DEFAULT_RECORD_LENGTH = 1024;
    static final long MAX_ENTRIES = Catalog.calculateMaxEntries(Integer.MAX_VALUE, 1024L);
    static final long DEFAULT_MAX_ENTRIES = 8192L;
    static final byte VALID = 1;
    static final byte INVALID = 0;
    private final RecordingDescriptorHeaderDecoder descriptorHeaderDecoder = new RecordingDescriptorHeaderDecoder();
    private final RecordingDescriptorHeaderEncoder descriptorHeaderEncoder = new RecordingDescriptorHeaderEncoder();
    private final RecordingDescriptorEncoder descriptorEncoder = new RecordingDescriptorEncoder();
    private final RecordingDescriptorDecoder descriptorDecoder = new RecordingDescriptorDecoder();
    private final CatalogHeaderDecoder catalogHeaderDecoder = new CatalogHeaderDecoder();
    private final MappedByteBuffer catalogByteBuffer;
    private final UnsafeBuffer catalogBuffer;
    private final UnsafeBuffer fieldAccessBuffer;
    private final int recordLength;
    private final int maxDescriptorStringsCombinedLength;
    private final int maxRecordingId;
    private final boolean forceWrites;
    private final boolean forceMetadata;
    private boolean isClosed;
    private final File archiveDir;
    private final EpochClock epochClock;
    private final FileChannel catalogChannel;
    private long nextRecordingId = 0L;

    Catalog(File archiveDir, FileChannel archiveDirChannel, int fileSyncLevel, long maxNumEntries, EpochClock epochClock) {
        this.archiveDir = archiveDir;
        this.forceWrites = fileSyncLevel > 0;
        this.forceMetadata = fileSyncLevel > 1;
        this.epochClock = epochClock;
        Catalog.validateMaxEntries(maxNumEntries);
        try {
            MappedByteBuffer catalogMappedByteBuffer;
            long catalogLength;
            File catalogFile = new File(archiveDir, "archive.catalog");
            boolean catalogPreExists = catalogFile.exists();
            FileChannel catalogFileChannel = null;
            try {
                catalogFileChannel = FileChannel.open(catalogFile.toPath(), StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.SPARSE);
                catalogLength = catalogPreExists ? Math.max(catalogFileChannel.size(), Catalog.calculateCatalogLength(maxNumEntries)) : Catalog.calculateCatalogLength(maxNumEntries);
                catalogMappedByteBuffer = catalogFileChannel.map(FileChannel.MapMode.READ_WRITE, 0L, catalogLength);
            }
            catch (Exception ex) {
                CloseHelper.close(catalogFileChannel);
                throw new RuntimeException(ex);
            }
            this.catalogChannel = catalogFileChannel;
            this.catalogByteBuffer = catalogMappedByteBuffer;
            this.catalogBuffer = new UnsafeBuffer(this.catalogByteBuffer);
            this.fieldAccessBuffer = new UnsafeBuffer(this.catalogByteBuffer);
            this.catalogHeaderDecoder.wrap(this.catalogBuffer, 0, 8, 0);
            if (catalogPreExists) {
                if (this.catalogHeaderDecoder.version() != 0) {
                    throw new IllegalArgumentException("catalog file version " + this.catalogHeaderDecoder.version() + " does not match software:" + 0);
                }
                this.recordLength = this.catalogHeaderDecoder.entryLength();
            } else {
                this.forceWrites(archiveDirChannel, this.forceWrites, this.forceMetadata);
                this.recordLength = 1024;
                new CatalogHeaderEncoder().wrap(this.catalogBuffer, 0).entryLength(1024).version(0);
            }
            this.maxDescriptorStringsCombinedLength = this.recordLength - 124;
            this.maxRecordingId = (int)Catalog.calculateMaxEntries(catalogLength, this.recordLength) - 1;
            this.refreshCatalog(true);
        }
        catch (Throwable ex) {
            this.close();
            throw ex;
        }
    }

    Catalog(File archiveDir, EpochClock epochClock) {
        this.archiveDir = archiveDir;
        this.forceWrites = false;
        this.forceMetadata = false;
        this.epochClock = epochClock;
        this.catalogChannel = null;
        try {
            MappedByteBuffer catalogMappedByteBuffer;
            long catalogLength;
            File catalogFile = new File(archiveDir, "archive.catalog");
            try (FileChannel channel = FileChannel.open(catalogFile.toPath(), StandardOpenOption.READ, StandardOpenOption.WRITE, StandardOpenOption.SPARSE);){
                catalogLength = channel.size();
                catalogMappedByteBuffer = channel.map(FileChannel.MapMode.READ_WRITE, 0L, catalogLength);
            }
            catch (Exception ex) {
                throw new RuntimeException(ex);
            }
            this.catalogByteBuffer = catalogMappedByteBuffer;
            this.catalogBuffer = new UnsafeBuffer(this.catalogByteBuffer);
            this.fieldAccessBuffer = new UnsafeBuffer(this.catalogByteBuffer);
            this.catalogHeaderDecoder.wrap(this.catalogBuffer, 0, 8, 0);
            if (this.catalogHeaderDecoder.version() != 0) {
                throw new IllegalArgumentException("catalog file version " + this.catalogHeaderDecoder.version() + " does not match software:" + 0);
            }
            this.recordLength = this.catalogHeaderDecoder.entryLength();
            this.maxDescriptorStringsCombinedLength = this.recordLength - 124;
            this.maxRecordingId = (int)Catalog.calculateMaxEntries(catalogLength, this.recordLength) - 1;
            this.refreshCatalog(false);
        }
        catch (Throwable ex) {
            this.close();
            throw ex;
        }
    }

    @Override
    public void close() {
        if (!this.isClosed) {
            this.isClosed = true;
            CloseHelper.close(this.catalogChannel);
            IoUtil.unmap(this.catalogByteBuffer);
        }
    }

    public int maxEntries() {
        return this.maxRecordingId + 1;
    }

    public int countEntries() {
        return (int)this.nextRecordingId;
    }

    public long addNewRecording(long startPosition, long startTimestamp, int imageInitialTermId, int segmentFileLength, int termBufferLength, int mtuLength, int sessionId, int streamId, String strippedChannel, String originalChannel, String sourceIdentity) {
        if (this.nextRecordingId > (long)this.maxRecordingId) {
            throw new ArchiveException("catalog is full, max recordings reached: " + this.maxEntries());
        }
        int combinedStringsLen = strippedChannel.length() + sourceIdentity.length() + originalChannel.length();
        if (combinedStringsLen > this.maxDescriptorStringsCombinedLength) {
            throw new ArchiveException("combined length of channel:'" + strippedChannel + "' and sourceIdentity:'" + sourceIdentity + "' and originalChannel:'" + originalChannel + "' exceeds max allowed:" + this.maxDescriptorStringsCombinedLength);
        }
        long recordingId = this.nextRecordingId++;
        this.catalogBuffer.wrap(this.catalogByteBuffer, this.recordingDescriptorOffset(recordingId), this.recordLength);
        this.descriptorEncoder.wrap(this.catalogBuffer, 32).recordingId(recordingId).startTimestamp(startTimestamp).stopTimestamp(-1L).startPosition(startPosition).stopPosition(-1L).initialTermId(imageInitialTermId).segmentFileLength(segmentFileLength).termBufferLength(termBufferLength).mtuLength(mtuLength).sessionId(sessionId).streamId(streamId).strippedChannel(strippedChannel).originalChannel(originalChannel).sourceIdentity(sourceIdentity);
        this.descriptorHeaderEncoder.wrap(this.catalogBuffer, 0).length(this.descriptorEncoder.encodedLength()).valid((byte)1);
        this.forceWrites(this.catalogChannel, this.forceWrites, this.forceMetadata);
        return recordingId;
    }

    public boolean wrapDescriptor(long recordingId, UnsafeBuffer buffer) {
        if (recordingId < 0L || recordingId > (long)this.maxRecordingId) {
            return false;
        }
        buffer.wrap(this.catalogByteBuffer, this.recordingDescriptorOffset(recordingId), this.recordLength);
        return Catalog.descriptorLength(buffer) > 0;
    }

    public boolean wrapAndValidateDescriptor(long recordingId, UnsafeBuffer buffer) {
        if (recordingId < 0L || recordingId > (long)this.maxRecordingId) {
            return false;
        }
        buffer.wrap(this.catalogByteBuffer, this.recordingDescriptorOffset(recordingId), this.recordLength);
        return Catalog.descriptorLength(buffer) > 0 && Catalog.isValidDescriptor(buffer);
    }

    public boolean hasRecording(long recordingId) {
        return recordingId >= 0L && recordingId < this.nextRecordingId && this.fieldAccessBuffer.getByte(this.recordingDescriptorOffset(recordingId) + RecordingDescriptorHeaderDecoder.validEncodingOffset()) == 1;
    }

    public void forEach(CatalogEntryProcessor consumer) {
        long recordingId = 0L;
        while (this.wrapDescriptor(recordingId, this.catalogBuffer)) {
            this.descriptorHeaderDecoder.wrap(this.catalogBuffer, 0, 32, 0);
            this.descriptorHeaderEncoder.wrap(this.catalogBuffer, 0);
            this.descriptorDecoder.wrap(this.catalogBuffer, 32, 80, 0);
            this.descriptorEncoder.wrap(this.catalogBuffer, 32);
            consumer.accept(this.descriptorHeaderEncoder, this.descriptorHeaderDecoder, this.descriptorEncoder, this.descriptorDecoder);
            ++recordingId;
        }
    }

    public boolean forEntry(CatalogEntryProcessor consumer, long recordingId) {
        if (this.wrapDescriptor(recordingId, this.catalogBuffer)) {
            this.descriptorHeaderDecoder.wrap(this.catalogBuffer, 0, 32, 0);
            this.descriptorHeaderEncoder.wrap(this.catalogBuffer, 0);
            this.descriptorDecoder.wrap(this.catalogBuffer, 32, 80, 0);
            this.descriptorEncoder.wrap(this.catalogBuffer, 32);
            consumer.accept(this.descriptorHeaderEncoder, this.descriptorHeaderDecoder, this.descriptorEncoder, this.descriptorDecoder);
            return true;
        }
        return false;
    }

    public long findLast(long minRecordingId, int sessionId, int streamId, byte[] channelFragment) {
        long recordingId = this.nextRecordingId;
        while (--recordingId >= minRecordingId) {
            this.catalogBuffer.wrap(this.catalogByteBuffer, this.recordingDescriptorOffset(recordingId), this.recordLength);
            if (!Catalog.isValidDescriptor(this.catalogBuffer)) continue;
            this.descriptorDecoder.wrap(this.catalogBuffer, 32, 80, 0);
            if (sessionId != this.descriptorDecoder.sessionId() || streamId != this.descriptorDecoder.streamId() || !Catalog.originalChannelContains(this.descriptorDecoder, channelFragment)) continue;
            return recordingId;
        }
        return -1L;
    }

    public static boolean originalChannelContains(RecordingDescriptorDecoder descriptorDecoder, byte[] channelFragment) {
        int offset;
        int fragmentLength = channelFragment.length;
        if (fragmentLength == 0) {
            return true;
        }
        int limit = descriptorDecoder.limit();
        int strippedChannelLength = descriptorDecoder.strippedChannelLength();
        int originalChannelOffset = limit + RecordingDescriptorDecoder.strippedChannelHeaderLength() + strippedChannelLength;
        descriptorDecoder.limit(originalChannelOffset);
        int channelLength = descriptorDecoder.originalChannelLength();
        descriptorDecoder.limit(limit);
        DirectBuffer buffer = descriptorDecoder.buffer();
        int end = offset + (channelLength - fragmentLength);
        block0: for (offset = descriptorDecoder.offset() + descriptorDecoder.sbeBlockLength() + RecordingDescriptorDecoder.strippedChannelHeaderLength() + strippedChannelLength + RecordingDescriptorDecoder.originalChannelHeaderLength(); offset <= end; ++offset) {
            for (int i = 0; i < fragmentLength; ++i) {
                if (buffer.getByte(offset + i) != channelFragment[i]) continue block0;
            }
            return true;
        }
        return false;
    }

    public void recordingStopped(long recordingId, long position, long timestamp) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32;
        long stopPosition = ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? position : Long.reverseBytes(position);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.stopTimestampEncodingOffset(), timestamp, RecordingDescriptorDecoder.BYTE_ORDER);
        this.fieldAccessBuffer.putLongVolatile(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), stopPosition);
        this.forceWrites(this.catalogChannel, this.forceWrites, this.forceMetadata);
    }

    public void recordingStopped(long recordingId, long position) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32;
        long stopPosition = ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? position : Long.reverseBytes(position);
        this.fieldAccessBuffer.putLongVolatile(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), stopPosition);
        this.forceWrites(this.catalogChannel, this.forceWrites, this.forceMetadata);
    }

    public void extendRecording(long recordingId, long controlSessionId, long correlationId) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32;
        long stopPosition = ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? -1L : Long.reverseBytes(-1L);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.controlSessionIdEncodingOffset(), controlSessionId);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.correlationIdEncodingOffset(), correlationId);
        this.fieldAccessBuffer.putLong(offset + RecordingDescriptorDecoder.stopTimestampEncodingOffset(), -1L);
        this.fieldAccessBuffer.putLongVolatile(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), stopPosition);
        this.forceWrites(this.catalogChannel, this.forceWrites, this.forceMetadata);
    }

    public long stopPosition(long recordingId) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32 + RecordingDescriptorDecoder.stopPositionEncodingOffset();
        long stopPosition = this.fieldAccessBuffer.getLongVolatile(offset);
        return ByteOrder.nativeOrder() == RecordingDescriptorDecoder.BYTE_ORDER ? stopPosition : Long.reverseBytes(stopPosition);
    }

    public RecordingSummary recordingSummary(long recordingId, RecordingSummary summary) {
        int offset = this.recordingDescriptorOffset(recordingId) + 32;
        summary.recordingId = recordingId;
        summary.startPosition = this.fieldAccessBuffer.getLong(offset + RecordingDescriptorDecoder.startPositionEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.stopPosition = this.fieldAccessBuffer.getLong(offset + RecordingDescriptorDecoder.stopPositionEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.initialTermId = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.initialTermIdEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.segmentFileLength = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.segmentFileLengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.termBufferLength = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.termBufferLengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.mtuLength = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.mtuLengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.streamId = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.streamIdEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        summary.sessionId = this.fieldAccessBuffer.getInt(offset + RecordingDescriptorDecoder.sessionIdEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
        return summary;
    }

    public static int descriptorLength(UnsafeBuffer descriptorBuffer) {
        return descriptorBuffer.getInt(RecordingDescriptorHeaderDecoder.lengthEncodingOffset(), RecordingDescriptorDecoder.BYTE_ORDER);
    }

    public static boolean isValidDescriptor(UnsafeBuffer descriptorBuffer) {
        return descriptorBuffer.getByte(RecordingDescriptorHeaderDecoder.validEncodingOffset()) == 1;
    }

    public static long calculateCatalogLength(long maxEntries) {
        return Math.min(maxEntries * 1024L + 1024L, Integer.MAX_VALUE);
    }

    public static long calculateMaxEntries(long catalogLength, long recordLength) {
        if (Integer.MAX_VALUE == catalogLength) {
            return (Integer.MAX_VALUE - (recordLength - 1L)) / recordLength;
        }
        return catalogLength / recordLength - 1L;
    }

    int recordingDescriptorOffset(long recordingId) {
        return (int)(recordingId * (long)this.recordLength) + this.recordLength;
    }

    public static void validateMaxEntries(long maxEntries) {
        if (maxEntries < 1L || maxEntries > MAX_ENTRIES) {
            throw new IllegalArgumentException("Catalog max entries must be between 1 and " + MAX_ENTRIES + ": maxEntries=" + maxEntries);
        }
    }

    public static long recoverStopOffset(File segmentFile, int segmentFileLength) {
        long lastFragmentOffset = 0L;
        try (FileChannel segment = FileChannel.open(segmentFile.toPath(), StandardOpenOption.READ);){
            int frameLength;
            ByteBuffer buffer = ByteBuffer.allocateDirect(32);
            buffer.order(RecordingDescriptorDecoder.BYTE_ORDER);
            long nextFragmentOffset = 0L;
            do {
                buffer.clear();
                if (32 != segment.read(buffer, nextFragmentOffset)) {
                    throw new ArchiveException("unexpected read failure from file: " + segmentFile.getAbsolutePath() + " at position:" + nextFragmentOffset);
                }
                frameLength = buffer.getInt(0);
                if (frameLength == 0) break;
                lastFragmentOffset = nextFragmentOffset;
            } while ((nextFragmentOffset += (long)BitUtil.align(frameLength, 32)) != (long)segmentFileLength);
            if (nextFragmentOffset / 4096L == lastFragmentOffset / 4096L) {
                lastFragmentOffset = nextFragmentOffset;
            }
        }
        catch (Exception ex) {
            LangUtil.rethrowUnchecked(ex);
        }
        return lastFragmentOffset;
    }

    private void refreshCatalog(boolean fixOnRefresh) {
        if (fixOnRefresh) {
            this.forEach(this::refreshAndFixDescriptor);
        } else {
            this.forEach((headerEncoder, headerDecoder, descriptorEncoder, descriptorDecoder) -> ++this.nextRecordingId);
        }
    }

    private void refreshAndFixDescriptor(RecordingDescriptorHeaderEncoder unused, RecordingDescriptorHeaderDecoder headerDecoder, RecordingDescriptorEncoder encoder, RecordingDescriptorDecoder decoder) {
        long recordingId = decoder.recordingId();
        if (headerDecoder.valid() == 1 && decoder.stopPosition() == -1L) {
            String prefix = recordingId + "-";
            String[] segmentFiles = this.archiveDir.list((dir, name) -> name.startsWith(prefix) && name.endsWith(".rec"));
            if (null == segmentFiles) {
                segmentFiles = ArrayUtil.EMPTY_STRING_ARRAY;
            }
            int maxSegmentIndex = -1;
            for (String filename : segmentFiles) {
                int offset;
                int length = filename.length();
                int remaining = length - (offset = prefix.length()) - ".rec".length();
                if (remaining <= 0) continue;
                try {
                    maxSegmentIndex = Math.max(AsciiEncoding.parseIntAscii(filename, offset, remaining), maxSegmentIndex);
                }
                catch (Exception exception) {
                    // empty catch block
                }
            }
            if (maxSegmentIndex < 0) {
                encoder.stopPosition(decoder.startPosition());
            } else {
                File maxSegmentFile = new File(this.archiveDir, Archive.segmentFileName(recordingId, maxSegmentIndex));
                int segmentFileLength = decoder.segmentFileLength();
                long stopOffset = Catalog.recoverStopOffset(maxSegmentFile, segmentFileLength);
                int termBufferLength = decoder.termBufferLength();
                long startPosition = decoder.startPosition();
                long recordingLength = (startPosition & (long)(termBufferLength - 1)) + (long)maxSegmentIndex * (long)segmentFileLength + stopOffset;
                encoder.stopPosition(startPosition + recordingLength);
            }
            encoder.stopTimestamp(this.epochClock.time());
        }
        this.nextRecordingId = recordingId + 1L;
    }

    private void forceWrites(FileChannel channel, boolean forceWrites, boolean forceMetadata) {
        if (null != channel && forceWrites) {
            try {
                channel.force(forceMetadata);
            }
            catch (Exception ex) {
                LangUtil.rethrowUnchecked(ex);
            }
        }
    }

    @FunctionalInterface
    static interface CatalogEntryProcessor {
        public void accept(RecordingDescriptorHeaderEncoder var1, RecordingDescriptorHeaderDecoder var2, RecordingDescriptorEncoder var3, RecordingDescriptorDecoder var4);
    }
}

