package com.ripple.cryptoconditions;

/*-
 * ========================LICENSE_START=================================
 * Crypto Conditions
 * %%
 * Copyright (C) 2016 - 2018 Ripple Labs
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =========================LICENSE_END==================================
 */

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * A helper class that can construct various types of Threshold Condition and Fulfillment.
 */
public class ThresholdFactory {

  //////////////////////
  // Condition Functions
  //////////////////////

  /**
   * <p>Constructs an 1-of-2 {@link ThresholdSha256Condition}, where the fulfillment of at least one
   * of the sub-conditions must be supplied in order to verify the generated threshold condition.</p>
   *
   * @param condition1 A sub-condition that satisfies the threshold requirement of the 1-of-2 condition generated by
   *                   this method.
   * @param condition2 Another sub-condition that satisfies the threshold requirement of the 1-of-2 condition generated
   *                   by this method.
   *
   * @return A newly created, immutable instance of {@link ThresholdSha256Condition} with a threshold of 1.
   */
  public static ThresholdSha256Condition oneOfTwoCondition(
      final Condition condition1, final Condition condition2
  ) {
    Objects.requireNonNull(condition1, "condition1 must not be null!");
    Objects.requireNonNull(condition2, "condition2 must not be null!");

    return constructMOfNCondition(
        1, 2, Stream.of(condition1, condition2).collect(Collectors.toList())
    );
  }

  /**
   * <p>Constructs an 2-of-2 {@link ThresholdSha256Condition}, where the fulfillment of both
   * sub-conditions must be supplied in order to verify against the threshold condition generated by this method.</p>
   *
   * @param condition1 A sub-condition that satisfies half of the threshold requirement of the 2-of-2 condition
   *                   generated by this method.
   * @param condition2 A sub-condition that satisfies the other half of the threshold requirement of the 2-of-2
   *                   condition generated by this method.
   *
   * @return A newly created, immutable instance of {@link ThresholdSha256Condition} with a threshold of 2.
   */
  public static ThresholdSha256Condition twoOfTwoCondition(
      final Condition condition1, final Condition condition2
  ) {
    Objects.requireNonNull(condition1, "condition1 must not be null!");
    Objects.requireNonNull(condition2, "condition2 must not be null!");

    return constructMOfNCondition(
        2, 2, Stream.of(condition1, condition2).collect(Collectors.toList())
    );
  }

  /**
   * <p>Construct an M-of-N threshold condition.</p>
   *
   * <p>For example, if M=3 and N=5, then this method will construct a 3-of-5 threshold condition
   * with five total sub-conditions, three of which must be fulfilled in order to fulfill the overall threshold
   * condition.</p>
   *
   * <p>Note that this method accepts a {@link List} of sub-conditions because it is permissible to
   * supply the same sub-condition more than once, for example for weighted threshold operations where one particular
   * sub-condition should be given more weight than another sub-condition.</p>
   *
   * <p>Concurrency Note: This method will create a shallow-copy of both {@code subconditions} and
   * {@code subfulfillments} before performing any operations, in order to guard against external list mutations. During
   * the brief period of time that this method is shallow-copying, callers should not consider this method to be
   * thread-safe. This is because another thread could mutate the lists (e.g., by adding or removing a sub-condition),
   * which may cause unpredictable behavior. In addition, this method assumes the sub-condition and sub-fulfillment
   * implementations are immutable, making shallow-copy operations sufficient to protect against external list mutation.
   * However, if your environment uses mutable implementations of either {@link Condition} or {@link Fulfillment}, such
   * shallow-copying may not be sufficient.</p>
   *
   * @param thresholdM       The minimum number of sub-fulfillments required to fulfill the threshold fulfillment
   *                         associated with the threshold condition generated by this method.
   * @param numSubCondtionsN The expected number of sub-conditions that this threshold condition can be verified
   *                         against.
   * @param subconditions    An {@link List} of sub-conditions (with size equal to N), a subset of which must be
   *                         fulfilled (size equals M) in order to fulfill the threshold fulfillment associated with the
   *                         threshold condition generated by this method.
   *
   * @return A newly constructed M-of-N Condition.
   */
  public static ThresholdSha256Condition constructMOfNCondition(
      final int thresholdM, int numSubCondtionsN, final List<Condition> subconditions
  ) {
    Objects.requireNonNull(subconditions, "subconditions must not be null!");

    if (thresholdM < 0) {
      throw new IllegalArgumentException("Threshold must not be negative!");
    }

    if (subconditions.size() != numSubCondtionsN) {
      throw new IllegalArgumentException(
          String.format("Number of sub-conditions must equal %d!", numSubCondtionsN));
    }
    return ThresholdSha256Condition.from(thresholdM, subconditions);
  }

  ////////////////////////
  // Fulfillment Functions
  ////////////////////////

  /**
   * <p>Constructs an 1-of-2 {@link ThresholdSha256Fulfillment}, where only 1 sub-fulfillment must
   * be supplied in order for the generated threshold fulfillment to verify properly.</p>
   *
   * @param fulfillment A sub-fulfillment that satisfies the threshold requirement of the fulfillment generated by this
   *                    method.
   * @param condition   An unfulfilled sub-conditions that when added to the condition derived from {@code fulfillment},
   *                    creates the list of all possible sub-conditions for the condition generated by this method.
   *
   * @return A newly created, immutable instance of {@link ThresholdSha256Fulfillment}.
   */
  public static ThresholdSha256Fulfillment oneOfTwoFulfillment(
      final Fulfillment fulfillment, final Condition condition
  ) {
    Objects.requireNonNull(condition, "condition must not be null!");
    Objects.requireNonNull(fulfillment, "fulfillment must not be null!");

    return constructMOfNFulfillment(1, 2, Collections.singletonList(condition),
        Collections.singletonList(fulfillment));
  }

  /**
   * <p>Constructs an 1-of-2 {@link ThresholdSha256Fulfillment}, where only 1 sub-fulfillment must
   * be supplied in order for the generated threshold fulfillment to verify properly.</p>
   *
   * @param fulfillment1 A sub-fulfillment that satisfies half of the threshold requirement of the fulfillment generated
   *                     by this method.
   * @param fulfillment2 A sub-fulfillment that satisfies the other half of the threshold requirement of the fulfillment
   *                     generated by this method.
   *
   * @return A newly created, immutable instance of {@link ThresholdSha256Fulfillment}.
   */
  public static ThresholdSha256Fulfillment twoOfTwoFulfillment(
      final Fulfillment fulfillment1, final Fulfillment fulfillment2
  ) {
    Objects.requireNonNull(fulfillment1, "fulfillment1 must not be null!");
    Objects.requireNonNull(fulfillment2, "fulfillment2 must not be null!");

    return constructMOfNFulfillment(2, 2,
        Collections.emptyList(), Stream.of(fulfillment1, fulfillment2).collect(Collectors.toList())
    );
  }

  /**
   * <p>Constructs an M-of-N {@link ThresholdSha256Fulfillment}, where M is the minimum number (or
   * threshold) of sub-fulfillments, and N is the combined total of sub-conditions plus sub-fulfillments. Note that
   * {@code totalN} must be supplied to this method and must match the combined total of {@code subconditions} and
   * {@code subfulfillments} in order for construction to work properly.</p>
   *
   * @param thresholdM      The minimum number of sub-fulfillments required to fulfill the threshold fulfillment
   *                        generated by this method.
   * @param totalN          The combined total number of sub-conditions and sub-fulfillments that this threshold
   *                        fulfillment can be verified against.
   * @param subconditions   An ordered {@link List} of unfulfilled sub-conditions that correspond to the threshold
   *                        condition being fulfilled. For example, if a given Threshold condition has 2 preimage
   *                        sub-conditions, but a threshold of 1, then a valid fulfillment would have one of those
   *                        preimage conditions in this List, and a fulfillment for the other preimage condition in the
   *                        {@code fulfillments} list. Note that this list must be combined with the list of conditions
   *                        derived from the subfulfillments, and the combined list, sorted, is used as the value when
   *                        deriving the fingerprint of this threshold fulfillment.
   * @param subfulfillments An ordered {@link List} of sub-fulfillments.  The number of elements in this list is equal
   *                        to the threshold of this fulfillment (i.e., per the crypto-condtions specification,
   *                        implementations must use the length of this list as the threshold value when deriving the
   *                        fingerprint of this crypto-condition).
   *
   * @return A newly created, immutable instance of {@link ThresholdSha256Fulfillment}.
   */
  public static ThresholdSha256Fulfillment constructMOfNFulfillment(
      final int thresholdM, int totalN,
      final List<Condition> subconditions, final List<Fulfillment> subfulfillments
  ) {
    Objects.requireNonNull(subconditions, "subconditions must not be null!");
    Objects.requireNonNull(subfulfillments, "subfulfillments must not be null!");

    if (thresholdM < 0) {
      throw new IllegalArgumentException("thresholdM (Threshold) must not be negative!");
    }

    final int computedM = subconditions.size() + subfulfillments.size();
    if (thresholdM > computedM) {
      throw new IllegalArgumentException(
          "Threshold must be less than or equal to the number of sub-fulfillments!");
    }

    // For a 3-of-5, fulfillments.size must be 3 and conditions.size must be 2 or fulfillments.size
    // must be 5 and conditions.size must be 0 because the implementation will derive conditions for
    // any supplied fulfillments (so callers must not supply a condition and an associated
    // fulfillment unless the condition should be represented twice (duplicated) on purposes, such
    // as in a weighted condition scenario.
    if (subconditions.size() + subfulfillments.size() != totalN) {
      throw new IllegalArgumentException(
          String.format("The combined number of sub-conditions and sub-fulfillments must equal %d!",
              totalN));
    }

    // For 1-of-2 fulfillment, if the threshold is set to 1, but 2 fulfillments are published,
    // then one of the fulfillments should be converted to a condition.

    // Create new, immutable lists
    final List<Condition> immutableSubconditions = Collections.unmodifiableList(
        // Skip the first _thresholdM_ sub-fulfillments, and use the rest of the sub-fulfillments
        // as sub-conditions, plus the originally supplied sub-conditions.
        Stream.concat(
            subconditions.stream(),
            subfulfillments.stream().skip(thresholdM).map(Fulfillment::getDerivedCondition)
        ).collect(Collectors.toList())
    );
    final List<Fulfillment> immutableSubFulfillments = Collections.unmodifiableList(
        // Use the first _thresholdM_ sub-fulfillments as the fulfillments.
        subfulfillments.stream().limit(thresholdM).collect(Collectors.toList())
    );

    return ThresholdSha256Fulfillment.from(immutableSubconditions, immutableSubFulfillments);
  }

}
