/*
 * Decompiled with CFR 0.152.
 */
package oshi.software.os.linux;

import com.sun.jna.Memory;
import com.sun.jna.Native;
import com.sun.jna.Pointer;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import oshi.jna.platform.linux.Libc;
import oshi.software.common.AbstractOperatingSystem;
import oshi.software.os.FileSystem;
import oshi.software.os.NetworkParams;
import oshi.software.os.OSProcess;
import oshi.software.os.OperatingSystem;
import oshi.software.os.linux.LinuxFileSystem;
import oshi.software.os.linux.LinuxNetworkParams;
import oshi.software.os.linux.LinuxOSVersionInfoEx;
import oshi.util.ExecutingCommand;
import oshi.util.FileUtil;
import oshi.util.ParseUtil;
import oshi.util.platform.linux.ProcUtil;

public class LinuxOperatingSystem
extends AbstractOperatingSystem {
    private static final long serialVersionUID = 1L;
    private static final Logger LOG = LoggerFactory.getLogger(LinuxOperatingSystem.class);
    protected String versionId;
    protected String codeName;
    private final int memoryPageSize;
    private static long hz = 1000L;
    private static long bootTime = 0L;

    public LinuxOperatingSystem() {
        this.manufacturer = "GNU/Linux";
        this.setFamilyFromReleaseFiles();
        this.version = new LinuxOSVersionInfoEx(this.versionId, this.codeName);
        this.memoryPageSize = LinuxOperatingSystem.getMemoryPageSize();
        LinuxOperatingSystem.init();
    }

    private static void init() {
        File[] pids = ProcUtil.getPidFiles();
        Arrays.sort(pids, (f1, f2) -> Integer.valueOf(f2.getName()).compareTo(Integer.valueOf(f1.getName())));
        long youngestJiffies = 0L;
        String youngestPid = "";
        for (File pid : pids) {
            String[] split;
            List<String> stat = FileUtil.readFile(String.format("/proc/%s/stat", pid.getName()));
            if (stat.isEmpty() || (split = stat.get(0).split("\\s+")).length < 22) continue;
            youngestJiffies = ParseUtil.parseLongOrDefault(split[21], 0L);
            youngestPid = pid.getName();
            break;
        }
        LOG.debug("Youngest PID is {} with {} jiffies", (Object)youngestPid, (Object)youngestJiffies);
        if (youngestJiffies == 0L) {
            LOG.error("Couldn't find any running processes, which is odd since we are in a running process. Process time values are in jiffies, not milliseconds.");
            return;
        }
        float startTimeSecsSinceBoot = ProcUtil.getSystemUptimeFromProc();
        bootTime = System.currentTimeMillis() - (long)(1000.0f * startTimeSecsSinceBoot);
        String etime = ExecutingCommand.getFirstAnswer(String.format("ps -p %s -o etimes=", youngestPid));
        if (!etime.isEmpty()) {
            LOG.debug("Etime is {} seconds", (Object)etime.trim());
            startTimeSecsSinceBoot -= Float.parseFloat(etime.trim());
        }
        LOG.debug("Start time in secs: {}", (Object)Float.valueOf(startTimeSecsSinceBoot));
        if (startTimeSecsSinceBoot <= 0.0f) {
            LOG.warn("Couldn't calculate jiffies per second. Process time values are in jiffies, not milliseconds.");
            return;
        }
        hz = (long)((float)youngestJiffies / startTimeSecsSinceBoot + 0.5f);
    }

    private static int getMemoryPageSize() {
        try {
            return Libc.INSTANCE.getpagesize();
        }
        catch (NoClassDefFoundError | UnsatisfiedLinkError e) {
            LOG.error("Failed to get the memory page size.", (Throwable)e);
            return 4096;
        }
    }

    @Override
    public FileSystem getFileSystem() {
        return new LinuxFileSystem();
    }

    @Override
    public OSProcess[] getProcesses(int limit, OperatingSystem.ProcessSort sort) {
        File[] pids;
        ArrayList<OSProcess> procs = new ArrayList<OSProcess>();
        for (File pid : pids = ProcUtil.getPidFiles()) {
            OSProcess proc = this.getProcess(ParseUtil.parseIntOrDefault(pid.getName(), 0));
            if (proc == null) continue;
            procs.add(proc);
        }
        List<OSProcess> sorted = this.processSort(procs, limit, sort);
        return sorted.toArray(new OSProcess[sorted.size()]);
    }

    @Override
    public OSProcess getProcess(int pid) {
        String[] split = FileUtil.getSplitFromFile(String.format("/proc/%d/stat", pid));
        if (split.length < 24) {
            return null;
        }
        String path = "";
        Memory buf = new Memory(1024L);
        int size = Libc.INSTANCE.readlink(String.format("/proc/%d/exe", pid), (Pointer)buf, 1023);
        if (size > 0) {
            path = buf.getString(0L).substring(0, size);
        }
        Map<String, String> io = FileUtil.getKeyValueMapFromFile(String.format("/proc/%d/io", pid), ":");
        long now = System.currentTimeMillis();
        OSProcess proc = new OSProcess();
        proc.setName(split[1].replaceFirst("\\(", "").replace(")", ""));
        proc.setPath(path);
        switch (split[2].charAt(0)) {
            case 'R': {
                proc.setState(OSProcess.State.RUNNING);
                break;
            }
            case 'S': {
                proc.setState(OSProcess.State.SLEEPING);
                break;
            }
            case 'D': {
                proc.setState(OSProcess.State.WAITING);
                break;
            }
            case 'Z': {
                proc.setState(OSProcess.State.ZOMBIE);
                break;
            }
            case 'T': {
                proc.setState(OSProcess.State.STOPPED);
                break;
            }
            default: {
                proc.setState(OSProcess.State.OTHER);
            }
        }
        proc.setProcessID(pid);
        proc.setParentProcessID(ParseUtil.parseIntOrDefault(split[3], 0));
        proc.setThreadCount(ParseUtil.parseIntOrDefault(split[19], 0));
        proc.setPriority(ParseUtil.parseIntOrDefault(split[17], 0));
        proc.setVirtualSize(ParseUtil.parseLongOrDefault(split[22], 0L));
        proc.setResidentSetSize(ParseUtil.parseLongOrDefault(split[23], 0L) * (long)this.memoryPageSize);
        proc.setKernelTime(ParseUtil.parseLongOrDefault(split[14], 0L) * 1000L / hz);
        proc.setUserTime(ParseUtil.parseLongOrDefault(split[13], 0L) * 1000L / hz);
        proc.setStartTime(bootTime + ParseUtil.parseLongOrDefault(split[21], 0L) * 1000L / hz);
        proc.setUpTime(now - proc.getStartTime());
        proc.setBytesRead(ParseUtil.parseLongOrDefault(io.getOrDefault("read_bytes", ""), 0L));
        proc.setBytesWritten(ParseUtil.parseLongOrDefault(io.getOrDefault("write_bytes", ""), 0L));
        List<String> stat = ExecutingCommand.runNative("stat -c %u,%U,%g,%G /proc/" + pid);
        if (!stat.isEmpty() && (split = stat.get(0).split(",")).length == 4) {
            proc.setUserID(split[0]);
            proc.setUser(split[1]);
            proc.setGroupID(split[2]);
            proc.setGroup(split[3]);
        }
        proc.setCommandLine(FileUtil.getStringFromFile(String.format("/proc/%d/cmdline", pid)));
        return proc;
    }

    @Override
    public int getProcessId() {
        return Libc.INSTANCE.getpid();
    }

    @Override
    public int getProcessCount() {
        return ProcUtil.getPidFiles().length;
    }

    @Override
    public int getThreadCount() {
        try {
            Libc.Sysinfo info = new Libc.Sysinfo();
            if (0 != Libc.INSTANCE.sysinfo(info)) {
                LOG.error("Failed to get process thread count. Error code: " + Native.getLastError());
                return 0;
            }
            return info.procs;
        }
        catch (NoClassDefFoundError | UnsatisfiedLinkError e) {
            LOG.error("Failed to get procs from sysinfo. {}", (Throwable)e);
            return 0;
        }
    }

    @Override
    public NetworkParams getNetworkParams() {
        return new LinuxNetworkParams();
    }

    private void setFamilyFromReleaseFiles() {
        if (this.family == null) {
            if (this.readOsRelease()) {
                return;
            }
            if (this.execLsbRelease()) {
                return;
            }
            if (this.readLsbRelease()) {
                return;
            }
            String etcDistribRelease = LinuxOperatingSystem.getReleaseFilename();
            if (this.readDistribRelease(etcDistribRelease)) {
                return;
            }
            this.family = LinuxOperatingSystem.filenameToFamily(etcDistribRelease.replace("/etc/", "").replace("release", "").replace("version", "").replace("-", "").replace("_", ""));
        }
    }

    private boolean readOsRelease() {
        if (new File("/etc/os-release").exists()) {
            List<String> osRelease = FileUtil.readFile("/etc/os-release");
            for (String line : osRelease) {
                if (line.startsWith("VERSION=")) {
                    LOG.debug("os-release: {}", (Object)line);
                    line = line.replace("VERSION=", "").replaceAll("^\"|\"$", "").trim();
                    String[] split = line.split("[()]");
                    if (split.length <= 1) {
                        split = line.split(", ");
                    }
                    if (split.length > 0) {
                        this.versionId = split[0].trim();
                    }
                    if (split.length <= 1) continue;
                    this.codeName = split[1].trim();
                    continue;
                }
                if (line.startsWith("NAME=") && this.family == null) {
                    LOG.debug("os-release: {}", (Object)line);
                    this.family = line.replace("NAME=", "").replaceAll("^\"|\"$", "").trim();
                    continue;
                }
                if (!line.startsWith("VERSION_ID=") || this.versionId != null) continue;
                LOG.debug("os-release: {}", (Object)line);
                this.versionId = line.replace("VERSION_ID=", "").replaceAll("^\"|\"$", "").trim();
            }
        }
        return this.family != null;
    }

    private boolean execLsbRelease() {
        for (String line : ExecutingCommand.runNative("lsb_release -a")) {
            if (line.startsWith("Description:")) {
                LOG.debug("lsb_release -a: {}", (Object)line);
                if (!(line = line.replace("Description:", "").trim()).contains(" release ")) continue;
                this.family = this.parseRelease(line, " release ");
                continue;
            }
            if (line.startsWith("Distributor ID:") && this.family == null) {
                LOG.debug("lsb_release -a: {}", (Object)line);
                this.family = line.replace("Distributor ID:", "").trim();
                continue;
            }
            if (line.startsWith("Release:") && this.versionId == null) {
                LOG.debug("lsb_release -a: {}", (Object)line);
                this.versionId = line.replace("Release:", "").trim();
                continue;
            }
            if (!line.startsWith("Codename:") || this.codeName != null) continue;
            LOG.debug("lsb_release -a: {}", (Object)line);
            this.codeName = line.replace("Codename:", "").trim();
        }
        return this.family != null;
    }

    private boolean readLsbRelease() {
        if (new File("/etc/lsb-release").exists()) {
            List<String> osRelease = FileUtil.readFile("/etc/lsb-release");
            for (String line : osRelease) {
                if (line.startsWith("DISTRIB_DESCRIPTION=")) {
                    LOG.debug("lsb-release: {}", (Object)line);
                    if (!(line = line.replace("DISTRIB_DESCRIPTION=", "").replaceAll("^\"|\"$", "").trim()).contains(" release ")) continue;
                    this.family = this.parseRelease(line, " release ");
                    continue;
                }
                if (line.startsWith("DISTRIB_ID=") && this.family == null) {
                    LOG.debug("lsb-release: {}", (Object)line);
                    this.family = line.replace("DISTRIB_ID=", "").replaceAll("^\"|\"$", "").trim();
                    continue;
                }
                if (line.startsWith("DISTRIB_RELEASE=") && this.versionId == null) {
                    LOG.debug("lsb-release: {}", (Object)line);
                    this.versionId = line.replace("DISTRIB_RELEASE=", "").replaceAll("^\"|\"$", "").trim();
                    continue;
                }
                if (!line.startsWith("DISTRIB_CODENAME=") || this.codeName != null) continue;
                LOG.debug("lsb-release: {}", (Object)line);
                this.codeName = line.replace("DISTRIB_CODENAME=", "").replaceAll("^\"|\"$", "").trim();
            }
        }
        return this.family != null;
    }

    private boolean readDistribRelease(String filename) {
        if (new File(filename).exists()) {
            List<String> osRelease = FileUtil.readFile(filename);
            for (String line : osRelease) {
                LOG.debug("{}: {}", (Object)filename, (Object)line);
                if (line.contains(" release ")) {
                    this.family = this.parseRelease(line, " release ");
                    break;
                }
                if (!line.contains(" VERSION ")) continue;
                this.family = this.parseRelease(line, " VERSION ");
                break;
            }
        }
        return this.family != null;
    }

    private String parseRelease(String line, String splitLine) {
        String[] split = line.split(splitLine);
        String family = split[0].trim();
        if (split.length > 1) {
            if ((split = split[1].split("[()]")).length > 0) {
                this.versionId = split[0].trim();
            }
            if (split.length > 1) {
                this.codeName = split[1].trim();
            }
        }
        return family;
    }

    protected static String getReleaseFilename() {
        File etc = new File("/etc");
        File[] matchingFiles = etc.listFiles(f -> (f.getName().endsWith("-release") || f.getName().endsWith("-version") || f.getName().endsWith("_release") || f.getName().endsWith("_version")) && !f.getName().endsWith("os-release") && !f.getName().endsWith("lsb-release"));
        if (matchingFiles != null && matchingFiles.length > 0) {
            return matchingFiles[0].getPath();
        }
        if (new File("/etc/release").exists()) {
            return "/etc/release";
        }
        return "/etc/issue";
    }

    private static String filenameToFamily(String name) {
        switch (name.toLowerCase()) {
            case "": {
                return "Solaris";
            }
            case "blackcat": {
                return "Black Cat";
            }
            case "bluewhite64": {
                return "BlueWhite64";
            }
            case "e-smith": {
                return "SME Server";
            }
            case "eos": {
                return "FreeEOS";
            }
            case "hlfs": {
                return "HLFS";
            }
            case "lfs": {
                return "Linux-From-Scratch";
            }
            case "linuxppc": {
                return "Linux-PPC";
            }
            case "meego": {
                return "MeeGo";
            }
            case "mandakelinux": {
                return "Mandrake";
            }
            case "mklinux": {
                return "MkLinux";
            }
            case "nld": {
                return "Novell Linux Desktop";
            }
            case "novell": 
            case "SuSE": {
                return "SUSE Linux";
            }
            case "pld": {
                return "PLD";
            }
            case "redhat": {
                return "Red Hat Linux";
            }
            case "sles": {
                return "SUSE Linux ES9";
            }
            case "sun": {
                return "Sun JDS";
            }
            case "synoinfo": {
                return "Synology";
            }
            case "tinysofa": {
                return "Tiny Sofa";
            }
            case "turbolinux": {
                return "TurboLinux";
            }
            case "ultrapenguin": {
                return "UltraPenguin";
            }
            case "va": {
                return "VA-Linux";
            }
            case "vmware": {
                return "VMWareESX";
            }
            case "yellowdog": {
                return "Yellow Dog";
            }
            case "issue": {
                return "Unknown";
            }
        }
        return name.substring(0, 1).toUpperCase() + name.substring(1);
    }

    static {
        LinuxOperatingSystem.init();
    }
}

